package provider

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/trust"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions/yapay"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/models"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/logic/payment"
	"errors"
	"sync"
)

type PaymentService struct {
	trustClient        trust.Client
	sandboxTrustClient trust.Client
	yapayClient        yapay.Client
	logger             log.Logger
	config             *Config
	resourceProvider   models.ResourceProvider
}

var _ payment.PaymentService = &PaymentService{}

func NewService(trustClient trust.Client, sandboxTrustClient trust.Client, yapayClient yapay.Client, logger log.Logger, config *Config, resourceProvider models.ResourceProvider) (*PaymentService, error) {
	if err := config.buildServiceByToken(); err != nil {
		return nil, err
	}

	return &PaymentService{trustClient: trustClient, sandboxTrustClient: sandboxTrustClient, yapayClient: yapayClient, logger: logger, config: config, resourceProvider: resourceProvider}, nil
}

func (s *PaymentService) getTrustClient(env models.OrderEnvironment) trust.Client {
	if env == models.SandboxOrder {
		return s.sandboxTrustClient
	}
	return s.trustClient
}

type Service struct {
	XServiceToken  string `config:"x_service_token" yaml:"x_service_token"`
	VerifyCvv      bool   `config:"verify_cvv" yaml:"verify_cvv"`
	CheckAntiFraud bool   `yaml:"check_anti_fraud"`
}

type Config struct {
	mu sync.RWMutex

	Services           []Service         `config:"services"`
	HideSBPQR          bool              `config:"hide_sbp_qr" yaml:"hide_sbp_qr"`
	DefaultAcquirer    string            `config:"default_acquirer,required" yaml:"default_acquirer"`
	YaPaymentsServices map[string]string `config:"ya_payments_acquirers,required" yaml:"ya_payments_acquirers"`
	LicenceURL         string            `config:"licence_url" yaml:"licence_url"`
	FamilyPay          struct {
		UnlimitedBalanceInFractionalUnits uint32 `config:"unlimited_balance_in_fractional_units,required" yaml:"unlimited_balance_in_fractional_units"`
	} `config:"family_pay,required" yaml:"family_pay"`
	GooglePay struct {
		Enabled bool `config:"enabled"`
		Gateway struct {
			Trust string `config:"trust"`
			YaPay string `config:"ya_pay" yaml:"ya_pay"`
		} `config:"gateway" yaml:"gateway"`
		PerUID     map[uint64]bool `config:"per_uid" yaml:"per_uid"`
		PerService map[string]bool `config:"per_service" yaml:"per_service"`
	} `config:"google_pay" yaml:"google_pay"`
	ApplePay struct {
		Enabled    bool            `config:"enabled"`
		PerUID     map[uint64]bool `config:"per_uid" yaml:"per_uid"`
		PerService map[string]bool `config:"per_service" yaml:"per_service"`
	} `config:"apple_pay" yaml:"apple_pay"`
	PartnerPaymentOptionsEnabled bool     `yaml:"partner_payment_options_enabled"`
	SandboxServiceToken          string   `yaml:"sandbox_service_token"`
	BoundPaymentMethodsWhitelist []string `yaml:"bound_payment_methods_whitelist"`

	serviceByToken map[string]*Service
}

func (c *Config) GetServiceToken(env models.OrderEnvironment, acquirer string) string {
	if env == models.SandboxOrder {
		return c.SandboxServiceToken
	}

	c.mu.RLock()
	defer c.mu.RUnlock()
	return c.YaPaymentsServices[acquirer]
}

func (c *Config) GetServiceTokenDefault(acquirer string) string {
	return c.GetServiceToken(models.ProductionOrder, acquirer)
}

func (c *Config) buildServiceByToken() error {
	c.mu.Lock()
	defer c.mu.Unlock()

	// Already built. Nothing to do here.
	if c.serviceByToken != nil {
		return nil
	}

	c.serviceByToken = make(map[string]*Service)
	for idx := range c.Services {
		service := &c.Services[idx]
		if _, prs := c.serviceByToken[service.XServiceToken]; prs {
			return errors.New("service X-Service-Token duplicate found")
		}
		c.serviceByToken[service.XServiceToken] = service
	}

	return nil
}

func (c *Config) GetServiceByToken(xServiceToken string) (*Service, error) {
	c.mu.RLock()
	defer c.mu.RUnlock()
	if c.serviceByToken == nil {
		if err := c.buildServiceByToken(); err != nil {
			return nil, err
		}
	}
	return c.serviceByToken[xServiceToken], nil
}
