package ctxutil

import (
	"a.yandex-team.ru/library/go/core/log"

	"github.com/blang/semver/v4"

	"context"
)

type key int

const (
	ctxStoreKey key = iota
)

type OptionalString struct {
	value string
	valid bool
}

type ctxStore struct {
	logger       log.Logger
	uid          uint64
	requestID    OptionalString
	userAgent    OptionalString
	sdkVersion   OptionalString
	remoteIP     OptionalString
	payToken     OptionalString
	serviceToken OptionalString
	path         OptionalString
	loginID      OptionalString
	forceCvv     bool
}

func getCtxStore(ctx context.Context) (*ctxStore, bool) {
	v := ctx.Value(ctxStoreKey)
	if v == nil {
		return &ctxStore{}, false
	}

	return v.(*ctxStore), true
}

func withCtxStore(ctx context.Context) (context.Context, *ctxStore) {
	store, ok := getCtxStore(ctx)
	if !ok {
		ctx = context.WithValue(ctx, ctxStoreKey, store)
	}
	return ctx, store
}

func WithLogger(ctx context.Context, logger log.Logger) context.Context {
	ctx, store := withCtxStore(ctx)
	store.logger = logger
	return ctx
}

func GetLogger(ctx context.Context) log.Logger {
	store, _ := getCtxStore(ctx)
	return store.logger
}

func WithRequestID(ctx context.Context, requestID string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.requestID = OptionalString{requestID, true}
	return ctx
}

func WithUserAgent(ctx context.Context, userAgent string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.userAgent = OptionalString{userAgent, true}
	return ctx
}

func WithSdkVersion(ctx context.Context, sdkVersion string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.sdkVersion = OptionalString{sdkVersion, true}
	return ctx
}

func WithRemoteIP(ctx context.Context, remoteIP string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.remoteIP = OptionalString{remoteIP, true}
	return ctx
}

func GetRequestID(ctx context.Context) string {
	store, _ := getCtxStore(ctx)
	return store.requestID.value
}

func WithUID(ctx context.Context, uid uint64) context.Context {
	ctx, store := withCtxStore(ctx)
	store.uid = uid
	return ctx
}

func GetUID(ctx context.Context) uint64 {
	store, _ := getCtxStore(ctx)
	return store.uid
}

func WithLoginID(ctx context.Context, loginID string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.loginID = OptionalString{loginID, true}
	return ctx
}

func GetLoginID(ctx context.Context) string {
	store, _ := getCtxStore(ctx)
	return store.loginID.value
}

func WithServiceToken(ctx context.Context, serviceToken string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.serviceToken = OptionalString{serviceToken, true}
	return ctx
}

func GetServiceToken(ctx context.Context) string {
	store, _ := getCtxStore(ctx)
	return store.serviceToken.value
}

func GetSdkVersion(ctx context.Context) string {
	store, _ := getCtxStore(ctx)
	return store.sdkVersion.value
}

func GetSdkSemverValue(ctx context.Context) (*semver.Version, error) {
	version := GetSdkVersion(ctx)
	return semver.New(version)
}

func WithPayToken(ctx context.Context, payToken string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.payToken = OptionalString{payToken, true}
	return ctx
}

func WithPath(ctx context.Context, path string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.path = OptionalString{path, true}
	return ctx
}

func WithForceCvv(ctx context.Context, forceCvv string) context.Context {
	ctx, store := withCtxStore(ctx)
	store.forceCvv = forceCvv == "1"
	return ctx
}

func GetForceCvv(ctx context.Context) bool {
	store, _ := getCtxStore(ctx)
	return store.forceCvv
}

func GetStoredFields(ctx context.Context) []log.Field {
	_, store := withCtxStore(ctx)
	result := make([]log.Field, 0, 5)
	if store.userAgent.valid {
		result = append(result, log.String("user_agent", store.userAgent.value))
	}
	if store.requestID.valid {
		result = append(result, log.String("request_id", store.requestID.value))
	}
	if store.sdkVersion.valid {
		result = append(result, log.String("sdk_version", store.sdkVersion.value))
	}
	if store.remoteIP.valid {
		result = append(result, log.String("remote_ip", store.remoteIP.value))
	}
	if store.payToken.valid {
		result = append(result, log.String("pay_token", store.payToken.value))
	}
	if store.uid > 0 {
		result = append(result, log.UInt64("uid", store.uid))
	}
	if store.path.valid {
		result = append(result, log.String("path", store.path.value))
	}
	if store.serviceToken.valid {
		result = append(result, log.String("service_token", store.serviceToken.value))
	}

	return result
}
