package stats

import (
	"a.yandex-team.ru/library/go/yandex/unistat"
	"a.yandex-team.ru/library/go/yandex/unistat/aggr"
	"fmt"
	"sync"
)

const statusCodePrefix = "response_status"
const timingPrefix = "response_time"
const defaultPriority = 0

var statusCodes = []int{200, 300, 400, 500}
var timingIntervals = []float64{0.0, .005, .01, .025, .05, .075, .1, .25, .5, .75, 1.0,
	2.5, 5.0, 7.5, 10.0, 12.0, 15.0, 20.0, 30.0}

type Metrics struct {
	mu              sync.Mutex
	name            string
	counters        map[string]*unistat.Numeric
	timingHistogram *unistat.Histogram
}

/*
	You must create Metrics object with specific name only once in application,
	since registry will raise error if there is a duplicated metric
*/
func NewMetrics(name string) *Metrics {
	counters := make(map[string]*unistat.Numeric)

	for _, sc := range statusCodes {
		key := counterKey(name, sc)
		metric := unistat.NewNumeric(key, defaultPriority, aggr.SummAlias{}, unistat.Sum)
		unistat.Register(metric)
		counters[key] = metric
	}
	timingHistogram := unistat.NewHistogram(fmt.Sprintf("%s_%s", timingPrefix, name), defaultPriority,
		aggr.HgramAlias{}, timingIntervals)
	unistat.Register(timingHistogram)

	return &Metrics{
		name:            name,
		counters:        counters,
		timingHistogram: timingHistogram,
	}
}

func (c *Metrics) UpdateRequestTime(rt float64) {
	c.timingHistogram.Update(rt)
}

func (c *Metrics) CountStatusCode(code int) {
	c.mu.Lock()
	defer c.mu.Unlock()

	metric := c.counters[counterKey(c.name, code)]
	if metric != nil {
		metric.Update(1)
	}
}

func counterKey(name string, code int) string {
	sCode := fmt.Sprintf("%dxx", code/100)
	return fmt.Sprintf("%s_%s_%s", statusCodePrefix, name, sCode)
}
