package tracing

import (
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/interactions"
	"a.yandex-team.ru/mail/payments-sdk-backend/internal/utils/ctxutil"
	"context"
	"fmt"
	"github.com/go-resty/resty/v2"
	"github.com/opentracing/opentracing-go"
	"github.com/opentracing/opentracing-go/log"
	"net/http"
	"strings"
)

const errorTag = "error"
const purchaseTokenTag = "purchase_token"

func SetOperationName(ctx context.Context, operationName string) {
	span := opentracing.SpanFromContext(ctx)
	if span == nil {
		return
	}

	span.SetOperationName(operationName)
}

func TagErrorWithMessage(ctx context.Context, message string, err error) {
	span := opentracing.SpanFromContext(ctx)
	if span == nil {
		return
	}

	span.SetTag(errorTag, true)
	span.LogFields(log.Message(message), log.Error(err))
}

func TagErrorMessage(ctx context.Context, message string) {
	span := opentracing.SpanFromContext(ctx)
	if span == nil {
		return
	}

	span.SetTag(errorTag, true)
	span.LogFields(log.Message(message))
}

func TagPurchaseToken(ctx context.Context, token string) {
	span := opentracing.SpanFromContext(ctx)
	if span == nil {
		return
	}

	span.SetTag(purchaseTokenTag, token)
}

func AddSpanToHTTPHeaders(ctx context.Context, headers *map[string]string) {
	span := opentracing.SpanFromContext(ctx)
	if span == nil {
		return
	}

	carrier := make(opentracing.HTTPHeadersCarrier)
	err := opentracing.GlobalTracer().Inject(span.Context(), opentracing.HTTPHeaders, carrier)
	if err != nil {
		logger := ctxutil.GetLogger(ctx)
		logger.Warn(fmt.Sprintf("Unable to inject jaeger context into http-headers: %s", err))
		return
	}
	for key, value := range carrier {
		if len(value) > 0 {
			(*headers)[key] = value[0]
		}
	}
}

func StartSpanForRequest(ctx context.Context, url string, request interactions.Request) (opentracing.Span, context.Context) {
	span, ctx := opentracing.StartSpanFromContext(
		ctx,
		"Outgoing HTTP request",
		opentracing.Tag{Key: "http.method", Value: request.Method},
		opentracing.Tag{Key: "http.path", Value: request.APIMethod},
		opentracing.Tag{Key: "http.url", Value: url},
	)
	AddSpanToHTTPHeaders(ctx, request.Headers)
	return span, ctx
}

func TracingDelegateRedirectPolicy(delegate resty.RedirectPolicy) resty.RedirectPolicy {
	return resty.RedirectPolicyFunc(func(req *http.Request, via []*http.Request) error {
		span, _ := opentracing.StartSpanFromContext(req.Context(), "Redirecting request",
			opentracing.Tag{Key: "http.from_path", Value: via[0].URL},
			opentracing.Tag{Key: "http.to_path", Value: req.URL},
		)
		defer span.Finish()
		return delegate.Apply(req, via)
	})
}

func SkipMonitoring(urlPath string) bool {
	return strings.HasSuffix(urlPath, "/ping") || strings.HasSuffix(urlPath, "/unistat")
}
