from typing import Any, Callable, Iterable, Optional

from aiohttp import web
from aiohttp.web_middlewares import normalize_path_middleware

from sendr_aiohttp import create_apispec, setup_swagger_route
from sendr_aiopg.engine.single import CustomEngine

from mail.payments.payments import __version__
from mail.payments.payments.api.middlewares import (
    middleware_header_cloud, middleware_logging_adapter, middleware_response_formatter, middleware_stats,
    middleware_tvm
)
from mail.payments.payments.api.routes.arbitrage import ARBITRAGE_ROUTES
from mail.payments.payments.api.routes.base import UrlDispatcher
from mail.payments.payments.api.routes.external import EXTERNAL_ROUTES
from mail.payments.payments.api.routes.internal import INTERNAL_ROUTES
from mail.payments.payments.api.routes.sdk import SDK_ROUTES
from mail.payments.payments.api.routes.testing import TESTING_ROUTES
from mail.payments.payments.api.routes.v1 import V1_ROUTES
from mail.payments.payments.conf import settings
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.storage.writers import PaymentsPushers
from mail.payments.payments.utils.db import create_configured_engine


class DBEngineApplication(web.Application):
    """
    Application that creates db_engine on startup and closes it on cleanup
    if no db_engine instance was provided
    """

    _db_engine: Optional[CustomEngine]

    @property
    def db_engine(self) -> CustomEngine:
        assert self._db_engine
        return self._db_engine

    def __init__(self, *args: Any, db_engine: Optional[CustomEngine] = None, **kwargs: Any) -> None:
        super().__init__(*args, **kwargs)
        if db_engine:
            self._db_engine = db_engine
        else:
            self._db_engine = None
            self.on_startup.append(self.open_engine)
            self.on_cleanup.append(self.close_engine)

    @staticmethod
    async def open_engine(app: 'DBEngineApplication') -> None:
        if not app._db_engine:
            app._db_engine = await create_configured_engine()

    @staticmethod
    async def close_engine(app: 'DBEngineApplication') -> None:
        if app._db_engine:
            app._db_engine.close()
            await app.db_engine.wait_closed()
            app._db_engine = None


class PaymentsApplication(DBEngineApplication):
    _urls = (
        EXTERNAL_ROUTES,
        INTERNAL_ROUTES,
        V1_ROUTES,
        SDK_ROUTES,
        ARBITRAGE_ROUTES,
        TESTING_ROUTES,
    )
    _middlewares: Iterable[Callable] = (
        normalize_path_middleware(remove_slash=True, append_slash=False),
        middleware_header_cloud,
        middleware_logging_adapter,
        middleware_stats,
        middleware_response_formatter,
        middleware_tvm,
    )

    def __init__(self, db_engine: Optional[CustomEngine] = None, crypto: Optional[Crypto] = None):
        self.crypto = crypto or Crypto.from_file(settings.CRYPTO_KEYS_FILE)
        self.pushers = PaymentsPushers()
        super().__init__(
            db_engine=db_engine,
            router=UrlDispatcher(),
            middlewares=self._middlewares,
        )
        self.add_routes()
        self.add_sentry()

        self.on_cleanup.append(self.close_interaction_connector)
        self.on_cleanup.append(self.pushers.close)

    def add_routes(self) -> None:
        for routes in self._urls:
            self.router.add_routes(routes)

        if settings.SWAGGER_ENABLED:
            spec = create_apispec(
                app=self,
                title='Яндекс.Оплата API',
                version=__version__,
            )
            setup_swagger_route(
                app=self,
                spec=spec,
                swagger_route='/api/swagger.json',
                swagger_route_name='swagger',
            )

    def add_sentry(self) -> None:
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            self.on_cleanup.append(sentry_init(settings.SENTRY_DSN, release=__version__))

    async def close_interaction_connector(self, _: web.Application) -> None:
        await AbstractInteractionClient.close_connector()
