from typing import Any, AsyncIterable, Optional, Type, Union

import ujson
from aiohttp import web
from marshmallow.exceptions import ValidationError

from sendr_aiohttp.handler import BaseHandler as BHandler
from sendr_aiohttp.handler import BaseParser
from sendr_qlog import LoggerContext
from sendr_tvm.qloud_async_tvm import TicketCheckResult

from mail.payments.payments.api.exceptions import APIException
from mail.payments.payments.api.schemas.base import BaseSchema, fail_response_schema
from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.core.entities.merchant_user import MerchantUser
from mail.payments.payments.core.exceptions import (
    BaseCoreError, CoreAlreadyExists, CoreDataError, CoreFieldError, CoreNotFoundError, CoreSecurityError
)


class Parser(BaseParser):
    def handle_error(self, error, *args, **kwargs):
        raise APIException(code=400, message='Bad Request', params=error.messages)


class BaseHandler(BHandler):
    PARSER = Parser()

    @property
    def app(self) -> web.Application:
        return self.request.app

    @property
    def logger(self) -> LoggerContext:
        return self.request['logger']

    @property
    def request_id(self) -> str:
        return self.request['request_id']

    @property
    def tvm(self) -> TicketCheckResult:
        return self.request['tvm']

    @property
    def user_ip(self) -> Optional[str]:
        return self.request.headers.get('X-REAL-IP', self.request.remote)

    def _core_exception_result(self, exc: BaseCoreError) -> None:
        code = 500
        message = getattr(exc, 'message', 'Internal server error')
        params = getattr(exc, 'params', {})
        if isinstance(exc, CoreNotFoundError):
            code = 404
        elif isinstance(exc, CoreAlreadyExists):
            code = 409
        elif isinstance(exc, CoreFieldError):
            message = 'Bad Request'
            code = 400
            params = params or {}
            if exc.fields:
                params.update(exc.fields)
        elif isinstance(exc, CoreDataError):
            code = 400
        elif isinstance(exc, CoreSecurityError):
            code = 403
        raise APIException(code=code, message=message, params=params)

    def get_params(self, params: Optional[dict] = None, schema: Optional[BaseSchema] = None) -> dict:
        if schema is not None:
            try:
                params, _ = schema.load(params)
            except ValidationError as error:
                raise APIException(code=400, message='Bad Request', params=error.messages)
        return params or {}

    @staticmethod
    def make_response(data: Union[dict, APIException],
                      schema: Optional[BaseSchema] = None,
                      *,
                      content_type: str = 'application/json',
                      headers: Optional[dict] = None,
                      status: int = 200,
                      ) -> web.Response:
        # TODO: надо убрать эту перегрузку.
        # Из-за неё мы указываем схему ответа дважды: в декораторе метода и в теле метода
        if schema is not None:
            data, _ = schema.dump(data)
        return web.Response(
            text=ujson.dumps(data),
            status=status,
            headers=headers,
            content_type=content_type,
        )

    async def make_stream_response(self,
                                   headers: dict,
                                   stream: AsyncIterable[bytes],
                                   ) -> web.StreamResponse:
        response = web.StreamResponse(headers=headers)
        await response.prepare(self.request)
        async for chunk in stream:
            await response.write(chunk)
        return response

    def make_error_response(self, data: dict) -> web.Response:
        return self.make_response(data=data, schema=fail_response_schema, status=400)

    def setup_action_context(self):
        """
        Устанавливаем специфичные task-local служебные переменные.
        Локальный контекст предзначен для инициализации/передачи низкоуровневых объектов приложения,
        необходимых для корректной работы кода приложения,
        и _НЕ_ предзначначен для передачи объектов, напрямую касающихся бизнес-логики.
        """
        BaseAction.context.logger = self.logger
        BaseAction.context.request_id = self.request_id
        BaseAction.context.db_engine = self.app.db_engine
        BaseAction.context.pushers = self.app.pushers
        BaseAction.context.crypto = self.app.crypto if hasattr(self.app, 'crypto') else None

    async def run_action(self, action_cls: Type[BaseAction], params: Optional[dict] = None) -> Any:
        self.setup_action_context()
        params = params or {}
        action = action_cls(**params)  # type: ignore
        try:
            return await action.run()
        except BaseCoreError as exc:
            self._core_exception_result(exc)


class BaseMerchantUserHandler(BaseHandler):
    """
    Хендлер требует аутентификации пользователя по юзер-тикету
    и наличия uid мерчанта в параметрах URL,
    для раннего связывания uid-ов "автора" запроса и мерчанта.

    Пару uid-ов помещаем в task-local контекстную переменную в виде MerchantUser объекта,
    для последующей авторизации действия пользователя на мерчанте.
    """

    def get_user_uid(self) -> Optional[int]:
        return self.tvm.default_uid

    def get_merchant_id(self) -> Optional[str]:
        return self.request.match_info['uid']

    def get_merchant_user(self) -> Optional[MerchantUser]:
        if not settings.CHECK_MERCHANT_USER:
            return None
        user_uid = self.get_user_uid()
        merchant_id = self.get_merchant_id()
        if user_uid is None or merchant_id is None:
            raise APIException(code=403, message='USER_NOT_AUTHENTICATED')

        return MerchantUser(user_uid=user_uid, merchant_id=merchant_id)

    def setup_action_context(self):
        super().setup_action_context()
        BaseAction.context.merchant_user = self.get_merchant_user()


class RequestMadeByUIDHandlerMixin:
    tvm: TicketCheckResult

    def request_is_made_by_uid(self, uid: int) -> bool:
        return self.tvm.default_uid == uid
