from sendr_aiohttp.handler import request_schema, response_schema

from mail.payments.payments.api.handlers.base import BaseHandler, BaseMerchantUserHandler
from mail.payments.payments.api.schemas.base import success_response_schema
from mail.payments.payments.api.schemas.internal.service import service_merchant_response_schema
from mail.payments.payments.api.schemas.mcc import mcc_request_schema, mcc_response_schema
from mail.payments.payments.api.schemas.merchant import (
    developer_key_schema, get_merchant_preregistration_response_schema, merchant_draft_request_schema,
    merchant_orders_stats_request_schema, merchant_orders_stats_response_schema, merchant_preregister_request_schema,
    merchant_request_schema, merchant_response_schema, merchant_service_list_response_schema,
    merchant_service_request_schema, merchant_suggest_request_schema, merchant_suggest_response_schema,
    merchant_token_response_schema, oauth_complete_request_schema, oauth_delete_request_schema,
    oauth_start_request_schema, oauth_start_response_schema
)
from mail.payments.payments.api.schemas.merchant_update import merchant_update_request_schema
from mail.payments.payments.api.schemas.moderation import ModerationRequestSchema
from mail.payments.payments.api.schemas.path import uid_request_schema, uid_service_merchant_id_schema
from mail.payments.payments.core.actions.merchant.create import CreateMerchantAction
from mail.payments.payments.core.actions.merchant.draft import CreateMerchantDraftAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.actions.merchant.get_by_key import GetMerchantByKeyAction
from mail.payments.payments.core.actions.merchant.get_mcc import GetMccInfoInTrustAction
from mail.payments.payments.core.actions.merchant.oauth_complete import OAuthCompleteMerchantAction
from mail.payments.payments.core.actions.merchant.oauth_delete import OAuthDeleteMerchantAction
from mail.payments.payments.core.actions.merchant.oauth_start import OAuthStartMerchantAction
from mail.payments.payments.core.actions.merchant.preregister import (
    GetMerchantPreregistrationAction, PreregisterMerchantAction
)
from mail.payments.payments.core.actions.merchant.stats import MerchantOrdersStatsAction
from mail.payments.payments.core.actions.merchant.suggest import MerchantSuggestAction
from mail.payments.payments.core.actions.merchant.token import GetMerchantTokenAction, RegenerateMerchantTokenAction
from mail.payments.payments.core.actions.merchant.update import UpdateMerchantAction
from mail.payments.payments.core.actions.moderation import ScheduleMerchantModerationAction
from mail.payments.payments.core.actions.service_merchant.delete import DeleteServiceMerchantAction
from mail.payments.payments.core.actions.service_merchant.get import (
    GetServiceMerchantAction, GetServiceMerchantListAction
)
from mail.payments.payments.core.actions.service_merchant.update import UpdateServiceMerchantAction
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.merchant_preregistration import MerchantPreregistration


class MerchantHandler(BaseMerchantUserHandler):
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_response_schema)
    async def get(self):
        merchant: Merchant = await self.run_action(GetMerchantAction, await self.get_data())
        return self.make_response({'data': merchant}, merchant_response_schema)

    @request_schema(merchant_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_response_schema)
    async def post(self):
        data = await self.run_action(CreateMerchantAction, await self.get_data())
        return self.make_response({'data': data}, merchant_response_schema)

    @request_schema(merchant_update_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_response_schema)
    async def patch(self):
        data = await self.run_action(UpdateMerchantAction, {'params': await self.get_data()})
        return self.make_response({'data': data}, merchant_response_schema)


class MerchantDraftHandler(BaseMerchantUserHandler):
    @request_schema(merchant_draft_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_response_schema)
    async def post(self):
        data = await self.run_action(CreateMerchantDraftAction, await self.get_data())
        return self.make_response({'data': data}, merchant_response_schema)


class MerchantPreregisterHandler(BaseMerchantUserHandler):
    @request_schema(merchant_preregister_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_response_schema)
    async def post(self):
        merchant: Merchant = await self.run_action(
            PreregisterMerchantAction,
            await self.get_data(),
        )
        return self.make_response({'data': merchant}, schema=merchant_response_schema)

    @request_schema(uid_request_schema, location='match_info')
    @response_schema(get_merchant_preregistration_response_schema)
    async def get(self):
        preregistration: MerchantPreregistration = await self.run_action(
            GetMerchantPreregistrationAction,
            await self.get_data(),
        )
        return self.make_response({'data': preregistration}, schema=get_merchant_preregistration_response_schema)


class MerchantByDeveloperKeyHandler(BaseHandler):
    @request_schema(developer_key_schema, location='match_info')
    @response_schema(merchant_response_schema)
    async def get(self):
        data = await self.run_action(GetMerchantByKeyAction, {
            **(await self.get_data()),
            'user_ip': self.user_ip,
        })
        return self.make_response({'data': data}, merchant_response_schema)


class MerchantModerationHandler(BaseMerchantUserHandler):
    @request_schema(uid_request_schema, location='match_info')
    @request_schema(ModerationRequestSchema(), location='json')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(ScheduleMerchantModerationAction, await self.get_data())
        return self.make_response({}, success_response_schema)


class MerchantTokenHandler(BaseMerchantUserHandler):
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_token_response_schema)
    async def get(self):
        token = await self.run_action(GetMerchantTokenAction, await self.get_data())
        return self.make_response({'data': token}, merchant_token_response_schema)

    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_token_response_schema)
    async def post(self):
        token = await self.run_action(RegenerateMerchantTokenAction, await self.get_data())
        return self.make_response({'data': token}, merchant_token_response_schema)


class MerchantServiceListHandler(BaseMerchantUserHandler):
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_service_list_response_schema)
    async def get(self):
        service_merchants = await self.run_action(GetServiceMerchantListAction, await self.get_data())
        return self.make_response(
            {'data': [{'service_merchant': service_merchant} for service_merchant in service_merchants]},
            merchant_service_list_response_schema
        )


class MerchantServiceHandler(BaseMerchantUserHandler):
    @request_schema(uid_service_merchant_id_schema, location='match_info')
    @response_schema(service_merchant_response_schema)
    async def get(self):
        service_merchant = await self.run_action(GetServiceMerchantAction, await self.get_data())
        return self.make_response({'data': {'service_merchant': service_merchant}}, service_merchant_response_schema)

    @request_schema(merchant_service_request_schema, location='json')
    @request_schema(uid_service_merchant_id_schema, location='match_info')
    @response_schema(service_merchant_response_schema)
    async def post(self):
        service_merchant = await self.run_action(UpdateServiceMerchantAction, await self.get_data())
        return self.make_response({'data': {'service_merchant': service_merchant}}, service_merchant_response_schema)

    @request_schema(uid_service_merchant_id_schema, location='match_info')
    @response_schema(success_response_schema)
    async def delete(self):
        await self.run_action(DeleteServiceMerchantAction, await self.get_data())
        return self.make_response({'data': {}}, success_response_schema)


class MerchantOAuthHandler(BaseMerchantUserHandler):
    @request_schema(oauth_delete_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(success_response_schema)
    async def delete(self):
        await self.run_action(OAuthDeleteMerchantAction, await self.get_data())
        return self.make_response({}, success_response_schema)


class MerchantOAuthStartHandler(BaseMerchantUserHandler):
    @request_schema(oauth_start_request_schema, location='query')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(oauth_start_response_schema)
    async def get(self):
        url = await self.run_action(OAuthStartMerchantAction, await self.get_data())
        return self.make_response({'data': {'url': url}}, oauth_start_response_schema)


class MerchantOAuthCompleteHandler(BaseMerchantUserHandler):
    @request_schema(oauth_complete_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(OAuthCompleteMerchantAction, await self.get_data())
        return self.make_response({}, success_response_schema)


class MccHandler(BaseHandler):
    @request_schema(uid_request_schema, location='match_info')
    @request_schema(mcc_request_schema, location='query')
    @response_schema(mcc_response_schema)
    async def get(self):
        data = await self.run_action(GetMccInfoInTrustAction, await self.get_data())
        return self.make_response({'data': data['result']}, mcc_response_schema)


class MerchantSuggestHandler(BaseHandler):
    @request_schema(merchant_suggest_request_schema, location='query')
    @response_schema(merchant_suggest_response_schema)
    async def get(self):
        data = await self.run_action(MerchantSuggestAction, await self.get_data())
        return self.make_response({'data': data}, merchant_suggest_response_schema)


class MerchantOrdersStatsHandler(BaseHandler):
    @request_schema(merchant_orders_stats_request_schema, location='query')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(merchant_orders_stats_response_schema)
    async def get(self):
        data = await self.run_action(MerchantOrdersStatsAction, await self.get_data())
        return self.make_response({'data': data}, merchant_orders_stats_response_schema)
