from sendr_aiohttp.handler import MethodSchema, request_schema, response_schema

from mail.payments.payments.api.handlers.base import BaseHandler, BaseMerchantUserHandler
from mail.payments.payments.api.schemas.base import success_response_schema
from mail.payments.payments.api.schemas.external.order import ref_request_schema
from mail.payments.payments.api.schemas.order import (
    active_order_request_schema, active_order_response_schema, get_order_list_request_schema,
    get_order_list_response_schema, get_order_response_schema, order_params_request_schema,
    pay_offline_order_response_schema, post_order_email_response_schema, post_order_email_schema,
    post_order_request_schema, post_order_response_schema, post_order_v1_clear_unhold_path_schema,
    post_pay_offline_order_request_schema, post_refund_customer_subscription_tx_schema, post_refund_request_schema,
    post_refund_response_schema, put_order_request_schema, put_order_response_schema
)
from mail.payments.payments.api.schemas.path import (
    customer_subscription_transaction_schema, order_active_uid_schema, order_uid_schema, shop_id_header_request_schema,
    uid_request_schema
)
from mail.payments.payments.core.actions.order.activate import ActivateOrderAction
from mail.payments.payments.core.actions.order.cancel import CancelOrderAction
from mail.payments.payments.core.actions.order.clear_unhold import ScheduleClearUnholdOrderAction
from mail.payments.payments.core.actions.order.create_from_multi import (
    CreateOrderFromMultiOrderAction, DownloadMultiOrderEmailListAction
)
from mail.payments.payments.core.actions.order.create_or_update import CreateOrUpdateOrderAction
from mail.payments.payments.core.actions.order.email import OrderEmailAction
from mail.payments.payments.core.actions.order.get import GetOrderAction
from mail.payments.payments.core.actions.order.get_list import GetOrderListAction
from mail.payments.payments.core.actions.order.pay_offline import PayOfflineOrderAction
from mail.payments.payments.core.actions.order.refund import (
    CreateCustomerSubscriptionTransactionRefundAction, CreateRefundAction
)
from mail.payments.payments.utils.helpers import iter_rows


class PutOrderMixin(BaseHandler):
    async def _get_or_create_method_schema(self) -> MethodSchema:
        request_method = self.request.method.lower()
        if request_method == 'put':
            method = await self._get_put_method()
            return MethodSchema.get_or_create_method_schema(method)
        return super()._get_or_create_method_schema()

    async def _get_put_method(self):
        try:
            data = await self.request.json()
        except ValueError:
            data = {}
        return self._active if 'active' in data else self._put

    async def put(self):
        method = await self._get_put_method()
        return await method()


class BaseOrderHandler(PutOrderMixin, BaseHandler):
    @request_schema(order_params_request_schema, location='query')
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(get_order_response_schema)
    async def get(self):
        params = {
            **await self.get_data(),
            'with_customer_subscription': True,
            'select_customer_subscription': None,
        }
        data = await self.run_action(GetOrderAction, params)
        return self.make_response({'data': data}, get_order_response_schema)

    @request_schema(active_order_request_schema, location='json')
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(active_order_response_schema)
    async def _active(self):
        self.logger.warning('Deprecated method for activate/deactivate order')
        data = await self.run_action(ActivateOrderAction, await self.get_data())
        return self.make_response({'data': data}, active_order_response_schema)

    @request_schema(put_order_request_schema, location='json')
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(put_order_response_schema)
    async def _put(self):
        data = await self.run_action(CreateOrUpdateOrderAction, await self.get_data())
        return self.make_response({'data': data}, put_order_response_schema)


class BaseOrderListHandler(BaseHandler):
    @request_schema(get_order_list_request_schema, location='query')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(get_order_list_response_schema)
    async def get(self):
        data = await self.run_action(GetOrderListAction, await self.get_data())
        return self.make_response({'data': data}, get_order_list_response_schema)

    @request_schema(post_order_request_schema, location='json')
    @request_schema(uid_request_schema, location='match_info')
    @request_schema(shop_id_header_request_schema, location='headers')
    @response_schema(post_order_response_schema)
    async def post(self):
        data = await self.run_action(CreateOrUpdateOrderAction, await self.get_data())
        return self.make_response({'data': data}, post_order_response_schema)


class BaseOrderActiveHandler(BaseHandler):
    @request_schema(order_active_uid_schema, location='match_info')
    @response_schema(post_order_response_schema)
    async def post(self):
        params = {
            **(await self.get_data()),
            'with_customer_subscription': True,
            'select_customer_subscription': None,
        }
        data = await self.run_action(ActivateOrderAction, params)
        return self.make_response({'data': data}, post_order_response_schema)


class OrderHandler(BaseMerchantUserHandler, BaseOrderHandler):
    pass


class OrderListHandler(BaseMerchantUserHandler, BaseOrderListHandler):
    pass


class OrderActiveHandler(BaseMerchantUserHandler, BaseOrderActiveHandler):
    pass


class OrderClearUnholdHandler(BaseMerchantUserHandler):
    @request_schema(post_order_v1_clear_unhold_path_schema, location='match_info')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(ScheduleClearUnholdOrderAction, await self.get_data())
        return self.make_response({'data': {}}, success_response_schema)


class OrderEmailHandler(BaseMerchantUserHandler):
    @request_schema(post_order_email_schema, location='json')
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(post_order_email_response_schema)
    async def post(self):
        params = {
            **(await self.get_data()),
            'user_ip': self.user_ip,
            'select_customer_subscription': None,
        }
        data = await self.run_action(OrderEmailAction, params)
        return self.make_response({'data': data}, post_order_email_response_schema)


class RefundHandler(BaseMerchantUserHandler):
    @request_schema(post_refund_request_schema, location='json')
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(post_refund_response_schema)
    async def post(self):
        data = await self.run_action(CreateRefundAction, await self.get_data())
        return self.make_response({'data': data}, post_refund_response_schema)


class RefundCustomerSubscriptionTransactionHandler(BaseMerchantUserHandler):
    @request_schema(post_refund_customer_subscription_tx_schema, location='json')
    @request_schema(customer_subscription_transaction_schema, location='match_info')
    @response_schema(post_refund_response_schema)
    async def post(self):
        data = await self.run_action(CreateCustomerSubscriptionTransactionRefundAction, await self.get_data())
        return self.make_response({'data': data}, post_refund_response_schema)


class CancelHandler(BaseMerchantUserHandler):
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(success_response_schema)
    async def post(self):
        """
        Отменить заказ, при условии что на нём нет активной (незавершенной) транзакции.
        Заказ нельзя будет оплатить после этого действия.
        """
        params = {
            **(await self.get_data()),
            'select_customer_subscription': None,
        }
        await self.run_action(CancelOrderAction, params)
        return self.make_response({'data': {}}, success_response_schema)


class MultiOrderHandler(BaseMerchantUserHandler):
    @request_schema(order_uid_schema, location='match_info')
    @response_schema(post_order_response_schema)
    async def post(self):
        data = await self.run_action(CreateOrderFromMultiOrderAction, await self.get_data())
        return self.make_response({'data': data}, post_order_response_schema)


class DownloadMultiOrderEmailListHandler(BaseMerchantUserHandler):
    @request_schema(order_uid_schema, location='match_info')
    async def get(self):
        rows = await self.run_action(DownloadMultiOrderEmailListAction, await self.get_data())
        headers = {
            'Content-Type': 'text/csv',
            'Content-Disposition': 'attachment; filename="email_list.csv"',
        }
        data = iter_rows(rows)
        return await self.make_stream_response(headers, data)


class PayOfflineOrderHandler(BaseMerchantUserHandler):
    @request_schema(ref_request_schema, location='query')
    @request_schema(order_uid_schema, location='match_info')
    @request_schema(post_pay_offline_order_request_schema, location='json')
    @response_schema(pay_offline_order_response_schema)
    async def post(self):
        data = await self.run_action(PayOfflineOrderAction, await self.get_data())
        return self.make_response({'data': data}, pay_offline_order_response_schema)
