from typing import Optional

from sendr_aiohttp.handler import request_schema, response_schema

from mail.payments.payments.api.exceptions import TVMUserTicketException
from mail.payments.payments.api.handlers.base import BaseHandler, BaseMerchantUserHandler, RequestMadeByUIDHandlerMixin
from mail.payments.payments.api.schemas.base import success_response_schema
from mail.payments.payments.api.schemas.path import merchant_id_request_schema, uid_request_schema
from mail.payments.payments.api.schemas.user_role import (
    delete_user_role_request_schema, merchant_user_role_list_response_schema, user_role_list_request_query_schema,
    user_role_list_response_schema, user_role_request_schema, user_role_response_schema
)
from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.user_role.create import CreateUserRoleAction
from mail.payments.payments.core.actions.user_role.delete import DeleteUserRoleAction
from mail.payments.payments.core.actions.user_role.get import GetUserRolesForMerchantAction, GetUserRolesForUserAction


class MerchantUserRoleHandler(BaseMerchantUserHandler):
    def get_merchant_id(self) -> Optional[str]:
        return self.request.match_info['merchant_id']

    @request_schema(user_role_list_request_query_schema, location='query')
    @request_schema(merchant_id_request_schema, location='match_info')
    @response_schema(merchant_user_role_list_response_schema)
    async def get(self):
        data = await self.run_action(GetUserRolesForMerchantAction, await self.get_data())
        return self.make_response({'data': data}, merchant_user_role_list_response_schema)

    @request_schema(user_role_request_schema, location='json')
    @request_schema(merchant_id_request_schema, location='match_info')
    @response_schema(user_role_response_schema)
    async def post(self):
        user_role = await self.run_action(CreateUserRoleAction, {**(await self.get_data()), 'notify': True})
        return self.make_response({'data': user_role}, user_role_response_schema)

    @request_schema(delete_user_role_request_schema, location='json')
    @request_schema(merchant_id_request_schema, location='match_info')
    @response_schema(success_response_schema)
    async def delete(self):
        await self.run_action(DeleteUserRoleAction, await self.get_data())
        return self.make_response({'data': {}}, success_response_schema)


class UserRoleHandler(RequestMadeByUIDHandlerMixin, BaseHandler):
    @request_schema(user_role_list_request_query_schema, location='query')
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(user_role_list_response_schema)
    async def get(self):
        """
        Показать все роли пользователя: список мерчантов и роль пользователя в них.
        Один uid может иметь ровно одну роль в мерчанте.
        Запрос должен быть подписан юзер-тикетом пользователя, для которого запрошен список ролей.
        """
        data = await self.get_data()
        uid = data.get('uid')
        if uid and (not self.request_is_made_by_uid(uid) and settings.CHECK_MERCHANT_USER):
            raise TVMUserTicketException
        data = await self.run_action(GetUserRolesForUserAction, data)
        return self.make_response({'data': data}, user_role_list_response_schema)
