from marshmallow import ValidationError, fields, post_load, pre_load
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.order import OrderSchema as BaseOrderSchema
from mail.payments.payments.core.entities.enums import ArbitrageStatus, ArbitrageVerdict


class EscalateIdSchema(BaseSchema):
    escalate_id = fields.String(load_from='conversationId', required=True)


class RefundItemArbitrageSchema(BaseSchema):
    item_id = fields.String(load_from='idInService', required=True)
    amount = fields.Decimal(required=True, load_from='quantity')

    # generated
    uid = fields.Integer()
    order_id = fields.Integer()
    product_id = fields.Integer()

    @pre_load
    def drop_amount(self, data: dict) -> dict:
        data.pop('amount', None)
        return data

    @post_load
    def ids(self, data: dict) -> dict:
        ids = data.pop('item_id').split('.')

        if len(ids) != 3:
            raise ValidationError('Invalid format.', ['idInService'])

        try:
            data['uid'], data['order_id'], data['product_id'] = list(map(int, ids))
        except ValueError:
            raise ValidationError('Invalid value.', ['idInService'])

        return data


class RefundArbitrageSchema(BaseSchema):
    items = fields.Nested(RefundItemArbitrageSchema, many=True, required=True)


class ArbitrageSchema(BaseSchema):
    arbitrage_id = fields.Integer()
    chat_id = fields.String()
    refund_id = fields.Integer()
    status = EnumField(ArbitrageStatus, by_value=True)
    created = fields.DateTime()
    updated = fields.DateTime()


class OrderSchema(BaseOrderSchema):
    is_create_arbitrage_available = fields.Boolean(dump_to='create_arbitrage_available')
    current_arbitrage = fields.Nested(ArbitrageSchema)


class GetOrderResponseSchema(SuccessResponseSchema):
    data = fields.Nested(OrderSchema)


class PostArbitrageConsultationResponseSchema(SuccessResponseSchema):
    data = fields.Nested(ArbitrageSchema)


class PostArbitrageVerdictRequestSchema(BaseSchema):
    verdict = EnumField(ArbitrageVerdict, by_value=True, load_from='type', required=True)
    refund = fields.Nested(RefundArbitrageSchema, required=False)

    # generated
    items = fields.List(fields.Dict())

    @pre_load
    def verdict_lower(self, data: dict) -> dict:
        data['verdict'] = str(data.get('type')).lower()
        return data

    @post_load
    def load_items(self, data: dict) -> dict:
        refund = data.pop('refund', None)
        if data['verdict'] == ArbitrageVerdict.REFUND:
            if not refund:
                raise ValidationError(fields.Field.default_error_messages['required'], ['refund'])
            data['items'] = refund['items']
        return data


escalate_id_schema = EscalateIdSchema()

get_order_response_schema = GetOrderResponseSchema()
post_arbitrage_consultation_response_schema = PostArbitrageConsultationResponseSchema()

post_arbitrage_verdict_request_schema = PostArbitrageVerdictRequestSchema()
