from collections.abc import Iterable, Mapping

import ujson
from marshmallow import Schema, SchemaOpts, fields, post_load, pre_dump, pre_load, validate

from mail.payments.payments.utils.helpers import without_none

CURRENCY_RUB = 'RUB'


class BaseSchema(Schema):
    skip_nones = True

    class Meta:
        json_module = ujson
        strict = True

    @pre_load
    def pre_load_skip_nones(self, data):
        # FIXME: Если вместо словаря в data окажется что-то ещё, то падаем
        return without_none(data) if self.skip_nones else data


class KeyNameFieldOpts(SchemaOpts):
    def __init__(self, meta):
        super().__init__(meta)
        self.key_name = getattr(meta, 'key_name', 'type')
        self.key_field = getattr(meta, 'key_field', fields.String())


class BaseListDictSchema(BaseSchema):
    """
    Deserializes dict of dicts into list of dicts.

    Example:
    ```
    class AddressesSchema(BaseListDictSchema):
        legal = fields.Nested(AddressSchema, required=True)
        post = fields.Nested(AddressSchema)

        class Meta:
            key_name = 'type'
            key_field = fields.String(validate=validate.OneOf(('legal', 'post')))
    ```
    This will deserialize this:
    {
        "legal": {...},
        "post": {...},
    }
    Into this:
    [
        {"type": "legal", ...},
        {"type": "post", ...},
    ]
    """
    OPTIONS_CLASS = KeyNameFieldOpts

    @pre_dump
    def from_list(self, data):
        if not isinstance(data, Iterable):
            self.fail('invalid')
        return {
            self.opts.key_field.serialize(self.opts.key_name, value): value
            for value in data
        }

    @post_load
    def to_list(self, data):
        if not isinstance(data, Mapping):
            self.fail('invalid')
        if not all((isinstance(value, Mapping) for value in data.values())):
            self.fail('invalid')
        return [
            {**value, self.opts.key_name: self.opts.key_field.deserialize(key)}
            for key, value in data.items()
        ]


class DefaultInsteadOfNoneMixin:
    """
    By default every field returns None if value is None.
    This mixin allows default to be serialized instead.
    """

    def _serialize(self, value, attr, obj):
        if value is None:
            value = self.default
        return super()._serialize(value, attr, obj)


class DefaultInsteadOfNoneNested(DefaultInsteadOfNoneMixin, fields.Nested):
    pass


class BasePaginatedRequestSchema(BaseSchema):
    limit = fields.Integer(missing=100, validate=validate.Range(min=1, max=100))
    offset = fields.Integer(missing=0, validate=validate.Range(min=0))


class BaseResponseSchema(BaseSchema):
    code = fields.Integer()
    status = fields.String()
    data: fields.Field = fields.Dict(default=None)


class SuccessResponseSchema(BaseResponseSchema):
    code = fields.Constant(200)
    status = fields.Constant('success')


class FailDataSchema(BaseSchema):
    message = fields.String()
    params = fields.Dict()


class FailResponseSchema(BaseResponseSchema):
    code = fields.Integer()
    status = fields.Constant('fail')
    data = fields.Nested(FailDataSchema)


success_response_schema = SuccessResponseSchema()
fail_response_schema = FailResponseSchema()
