from marshmallow import ValidationError, fields, post_load, validate
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.fields import StripString
from mail.payments.payments.api.schemas.order import PayTokenOrderSchema, PostOrderRequestSchema, PutOrderRequestSchema
from mail.payments.payments.core.entities.enums import NDS, PAYMETHOD_ID_OFFLINE, PaymentsTestCase, ReceiptType


class ResizedItemSchema(BaseSchema):
    amount = fields.Decimal(places=2, required=True)
    price = fields.Decimal(places=2, required=True)
    product_id = fields.Integer(required=True)


class ReceiptCloseSchema(BaseSchema):
    nds = EnumField(NDS, by_value=True)
    product_id = fields.Integer(required=True)


class InternalOrderRequestSchema(BaseSchema):
    with_refunds = fields.Boolean(default=False)


class InternalOrderResponseSchema(SuccessResponseSchema):
    data = fields.Nested(PayTokenOrderSchema)


class InternalOrderListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(PayTokenOrderSchema, many=True)


class PostInternalOrderRequestSchema(PostOrderRequestSchema):
    commission = fields.Integer(allow_none=True)
    customer_uid = fields.Integer()
    user_email = StripString()
    user_description = fields.String()
    test = EnumField(PaymentsTestCase, by_value=True, allow_none=True)
    receipt_type = EnumField(ReceiptType, by_value=True)
    paymethod_id = fields.String(validate=validate.NoneOf([PAYMETHOD_ID_OFFLINE]))
    service_data = fields.Dict(allow_none=True)
    recurrent = fields.Boolean(default=False)
    without_3ds = fields.Boolean(default=False)

    @post_load
    def validate_order(self, data):
        if data.get('recurrent') and data.get('without_3ds'):
            raise ValidationError('Order cannot combine both recurrent and without_3ds properties')

        return data


class PostInternalOrderStartRequestSchema(BaseSchema):
    yandexuid = fields.String(allow_none=True)
    login_id = fields.String(allow_none=True)

    email = StripString()
    return_url = fields.String()
    customer_uid = fields.Integer(missing=None, allow_none=True)
    description = fields.String()
    template = fields.String(validate=validate.OneOf(['mobile', 'desktop']))
    trust_form_name = fields.String()
    service_data = fields.Dict(allow_none=True)
    payment_completion_action = fields.Field()  # PAYBACK-646

    @post_load
    def post_load(self, data):
        payment_completion_action = data.get('payment_completion_action')

        if isinstance(payment_completion_action, dict):
            for k, v in payment_completion_action.items():
                if not isinstance(k, str) or not isinstance(v, str):
                    raise ValidationError('Keys and values must be strings.', ['payment_completion_action'])
        elif payment_completion_action and not isinstance(payment_completion_action, str):
            raise ValidationError('Field must be string or dict.', ['payment_completion_action'])

        return data


class PostInternalPayOfflineOrderRequestSchema(BaseSchema):
    customer_uid = fields.Integer(missing=None, allow_none=True)
    description = fields.String()


class PaymentStatusSchema(BaseSchema):
    trust_url = fields.String(required=True, attribute='payment_url')


class PostInternalOrderStartResponseSchema(SuccessResponseSchema):
    data = fields.Nested(PaymentStatusSchema)


class PutInternalOrderRequestSchema(PutOrderRequestSchema):
    class Meta:
        strict = True
        fields = PutOrderRequestSchema.Meta.fields + ('service_data', 'commission')

    service_data = fields.Dict(allow_none=True)
    commission = fields.Integer(allow_none=True)


class PostInternalUpdateServiceDataSchema(BaseSchema):
    service_data = fields.Dict(allow_none=True)


class PostResizedOrderRequestSchema(BaseSchema):
    items = fields.Nested(ResizedItemSchema, many=True)


class PostInternalReceiptCloseRequestSchema(BaseSchema):
    items = fields.Nested(ReceiptCloseSchema, many=True, missing=None, allow_none=True)


class PayoutSchema(BaseSchema):
    status = fields.String()
    payment_batch_id = fields.Integer()
    payment_number = fields.String()
    update_dt = fields.String()
    payment_status = fields.String()
    batch_payment_amount = fields.String()
    dt = fields.String()
    payment_details = fields.String()


class GetInternalOrderPayoutInfoSchema(SuccessResponseSchema):
    data = fields.Nested(PayoutSchema, many=True)


get_internal_order_request_schema = InternalOrderRequestSchema()
get_internal_order_response_schema = InternalOrderResponseSchema()

get_internal_order_list_response_schema = InternalOrderListResponseSchema()
post_internal_order_response_schema = InternalOrderResponseSchema()

put_internal_order_request_schema = PutInternalOrderRequestSchema()
put_internal_order_response_schema = InternalOrderResponseSchema()

active_internal_order_response_schema = InternalOrderResponseSchema()

post_internal_pay_offline_order_request_schema = PostInternalPayOfflineOrderRequestSchema()
post_internal_pay_offline_order_response_schema = InternalOrderResponseSchema()

post_internal_order_request_schema = PostInternalOrderRequestSchema()

post_internal_order_start_request_schema = PostInternalOrderStartRequestSchema()
post_internal_order_start_response_schema = PostInternalOrderStartResponseSchema()

post_internal_order_resized_request_schema = PostResizedOrderRequestSchema()

post_internal_update_service_data_request_schema = PostInternalUpdateServiceDataSchema()
post_internal_update_service_data_response_schema = InternalOrderResponseSchema()

post_internal_receipt_close_request_schema = PostInternalReceiptCloseRequestSchema()

get_internal_order_payout_info_schema = GetInternalOrderPayoutInfoSchema()
