import re
import string
from dataclasses import asdict
from datetime import datetime
from typing import Any

from marshmallow import ValidationError, fields, missing, post_load, pre_dump, pre_load, validate
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BaseListDictSchema, BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.fields import StripString
from mail.payments.payments.api.schemas.functionality import FunctionalityAPISchema
from mail.payments.payments.api.schemas.moderation import ModerationSchema, ModerationsSchema
from mail.payments.payments.api.schemas.path import UIDRequestSchema
from mail.payments.payments.core.entities.enums import (
    PAY_METHODS, AcquirerType, FunctionalityType, GraphType, GroupType, MerchantOAuthMode, MerchantStatus, MerchantType,
    PersonType, RegistrationRoute
)
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.not_fetched import NOT_FETCHED
from mail.payments.payments.utils.const import MAX_LENGTH_ENGLISH_NAME
from mail.payments.payments.utils.helpers import is_entrepreneur_by_inn


class AddressSchema(BaseSchema):
    ZIP_REGEXP = '^([0-9]){6}$'

    city = fields.String(required=True, validate=validate.Length(1, 28))
    country = fields.String(required=True, validate=validate.OneOf(('RUS',)))
    home = fields.String(required=True)
    street = fields.String(required=True)
    zip = fields.String(required=True, validate=validate.Regexp(ZIP_REGEXP))


class PartialAddressSchema(AddressSchema):
    pass


class AddressesSchema(BaseListDictSchema):
    legal = fields.Nested(AddressSchema, required=True)
    post = fields.Nested(AddressSchema, allow_none=True)

    class Meta:
        key_name = 'type'
        key_field = fields.String(validate=validate.OneOf(('legal', 'post')))


class AddressesPartialSchema(AddressesSchema):
    legal = fields.Nested(PartialAddressSchema(partial=True))
    post = fields.Nested(PartialAddressSchema(partial=True))


class BankSchema(BaseSchema):
    CORR_ACCOUNT_REGEXP = '^[0-9]+$'
    BIK_REGEXP = '^[0-9]+$'

    account = fields.String(required=True)
    bik = fields.String(required=True, validate=validate.Regexp(BIK_REGEXP))
    correspondent_account = fields.String(
        dump_to='correspondentAccount',
        load_from='correspondentAccount',
        required=True,
        validate=validate.Regexp(CORR_ACCOUNT_REGEXP),
    )
    name = fields.String(required=True)


class BillingSchema(BaseSchema):
    client_id = fields.String()
    person_id = fields.String()
    contract_id = fields.String()
    partner_id = fields.String(attribute='client_id', dump_to='trust_partner_id')
    submerchant_id = fields.String(dump_to='trust_submerchant_id')


class InnFieldValidators:
    MAX_LENGTH = 32
    REGEXP = '^[0-9]+$'

    _validators = (validate.Length(1, MAX_LENGTH), validate.Regexp(REGEXP))

    def __iter__(self):
        return iter(self._validators)


class OrganizationSchema(BaseSchema):
    MAX_LENGTH_NAME = 512
    OGRN_REGEXP = '^[0-9]+$'
    ENGLISH_NAME_REGEXP = '[A-z0-9.\\-_ ]+'
    INN_REGEXP = '^[0-9]+$'
    SITE_URL = '^[^\\s]*$'

    type = EnumField(MerchantType, required=True, by_value=True)
    name = fields.String(allow_none=True, validate=validate.Length(1, MAX_LENGTH_NAME))
    english_name = fields.String(dump_to='englishName',
                                 load_from='englishName',
                                 allow_none=True,
                                 validate=[validate.Length(1, MAX_LENGTH_ENGLISH_NAME),
                                           validate.Regexp(ENGLISH_NAME_REGEXP)])
    full_name = fields.String(dump_to='fullName',
                              load_from='fullName',
                              allow_none=True,
                              validate=validate.Length(1))
    inn = fields.String(required=True, validate=InnFieldValidators())
    kpp = fields.String(missing=None, allow_none=True, validate=validate.Regexp(INN_REGEXP))
    ogrn = fields.String(required=True, validate=validate.Regexp(OGRN_REGEXP))
    schedule_text = fields.String(dump_to='scheduleText',
                                  load_from='scheduleText',
                                  allow_none=True,
                                  missing=None)
    site_url = fields.String(dump_to='siteUrl',
                             load_from='siteUrl',
                             validate=validate.Regexp(SITE_URL))
    description = fields.String(allow_none=True)

    @post_load
    def post_load_check_organization_fields(self, data):
        if self.partial or is_entrepreneur_by_inn(data['inn']):
            return

        for field_name in ('name', 'english_name', 'full_name'):
            if data.get(field_name) is None:
                raise ValidationError('Missing field.', [field_name])


class PersonSchema(BaseSchema):
    name = fields.String(required=True)
    email = fields.String(required=True, validate=validate.Email())
    phone = fields.String(required=True)
    surname = fields.String(required=True)
    patronymic = fields.String(allow_none=True)
    birth_date = fields.Date(dump_to='birthDate', load_from='birthDate')

    @pre_dump
    def pre_dump(self, data):
        if isinstance(data, dict) and data.get('birth_date') is not None and isinstance(data.get('birth_date'), str):
            data['birth_date'] = datetime.fromisoformat(data['birth_date']).date()
        return data


class PartialPersonSchema(PersonSchema):
    pass


class PersonsSchema(BaseListDictSchema):
    ceo = fields.Nested(PersonSchema, required=True)
    contact = fields.Nested(PersonSchema, allow_none=True)
    signer = fields.Nested(PersonSchema, allow_none=True)

    class Meta:
        key_name = 'type'
        key_field = EnumField(PersonType, by_value=True)


class PersonsPartialSchema(PersonsSchema):
    ceo = fields.Nested(PartialPersonSchema(partial=True))
    contact = fields.Nested(PartialPersonSchema(partial=True))
    signer = fields.Nested(PartialPersonSchema(partial=True))


class OAuthSchema(BaseSchema):
    account_id = fields.Integer(attribute='data.account_id')
    expired = fields.Boolean()
    mode = EnumField(MerchantOAuthMode, by_value=True)


class MerchantOptionsSchema(BaseSchema):
    allow_create_orders_via_ui = fields.Boolean()
    allow_create_service_merchants = fields.Boolean()
    hide_commission = fields.Boolean()
    can_skip_registration = fields.Boolean()


class DumpPreregisterDataSchema(BaseSchema):
    inn = fields.String(required=True, allow_none=True, dump_only=True)
    services = fields.List(fields.Integer(), required=True, allow_none=True, dump_only=True)
    require_online = fields.Boolean(required=True, allow_none=True, dump_only=True)
    categories = fields.List(fields.Integer(), required=True, allow_none=True, dump_only=True)


class CreateMerchantSchema(BaseSchema):
    name = fields.String(required=True)
    username = fields.String(missing=None, allow_none=True)
    addresses = fields.Nested(AddressesSchema, required=True)
    bank = fields.Nested(BankSchema, required=True)
    organization = fields.Nested(OrganizationSchema, required=True)
    persons = fields.Nested(PersonsSchema, required=True)
    fast_moderation = fields.Boolean(missing=False)
    functionality = fields.Nested(FunctionalityAPISchema, missing={'type': FunctionalityType.PAYMENTS.value})


class CreateMerchantDraftSchema(BaseSchema):
    class Meta:
        strict = True

    name = fields.String(required=True)
    username = fields.String()
    addresses = fields.Nested(AddressesPartialSchema)
    bank = fields.Nested(BankSchema(partial=True))
    organization = fields.Nested(OrganizationSchema(partial=True))
    persons = fields.Nested(PersonsPartialSchema)


class FunctionalitiesSchema(BaseSchema):
    payments = fields.Nested(FunctionalityAPISchema)
    yandex_pay = fields.Nested(FunctionalityAPISchema)


class DumpMerchantSchema(BaseSchema):
    class Meta:
        strict = True

    name = fields.String(required=True, dump_only=True)
    username = fields.String(dump_only=True)
    registered = fields.Boolean(dump_only=True)
    addresses = fields.Nested(AddressesPartialSchema, dump_only=True)
    bank = fields.Nested(BankSchema(partial=True), dump_only=True)
    organization = fields.Nested(OrganizationSchema(partial=True), dump_only=True)
    persons = fields.Nested(PersonsPartialSchema, dump_only=True)
    fast_moderation = fields.Boolean(dump_only=True)

    uid = fields.Integer(dump_only=True)
    merchant_id = fields.String(allow_none=True, dump_only=True)
    created = fields.DateTime(dump_only=True)
    parent_uid = fields.Integer(dump_only=True)
    revision = fields.Integer(dump_only=True)
    status = EnumField(MerchantStatus, by_value=True, dump_only=True)
    updated = fields.DateTime(dump_only=True)
    acquirer = EnumField(AcquirerType, by_value=True, dump_only=True)

    allow_create_orders_via_ui = fields.Boolean(dump_only=True)
    options = fields.Nested(MerchantOptionsSchema, dump_only=True)
    registration_route = EnumField(RegistrationRoute, by_value=True, dump_only=True)

    # fields excluded when dumping Merchant in draft status
    billing = fields.Nested(BillingSchema, dump_only=True)
    oauth = fields.Nested(OAuthSchema, dump_only=True, many=True)
    moderation = fields.Nested(ModerationSchema, dump_only=True)
    moderations = fields.Nested(ModerationsSchema, dump_only=True)
    preregister_data = fields.Nested(DumpPreregisterDataSchema, dump_only=True)
    functionalities = fields.Nested(FunctionalitiesSchema, dump_only=True)

    def get_attribute(self, name: str, obj: Merchant, default_value: Any) -> Any:
        """
        Проксируем часть полей для мерчантов-черновиков.
        Делаем вид, что каких-то полей нет в данных, отдавая missing.
        """

        if obj.status == MerchantStatus.DRAFT:
            if name in ('bank', 'organization', 'username', 'registered', 'persons', 'addresses', 'fast_moderation'):
                draft_data: dict = obj.draft_data or {}
                value = draft_data.get(name)

                if name in ('registered', 'fast_moderation'):
                    value = bool(value)
                elif value is not None:
                    if name == 'persons':
                        for person in value:
                            person['type'] = PersonType(person['type'])
                    elif name == 'organization':
                        value['type'] = MerchantType(value['type'])

                return value
            elif name in ('moderation', 'oauth', 'billing', 'preregister_data'):
                # так же для драфтов считаем, что нет данных полей
                return missing

        # expecting merchant data IS loaded
        if name == 'billing':
            return dict(
                client_id=obj.client_id,
                person_id=obj.person_id,
                contract_id=obj.contract_id,
                submerchant_id=obj.submerchant_id,
            )

        if name == 'oauth' and obj.oauth is NOT_FETCHED:
            # если не извлечено - сделаем вид, что его нет
            return missing

        if name == 'preregister_data' and obj.preregistration is None:
            return missing

        return super().get_attribute(name, obj, default_value)


class CreateMerchantPreregisterSchema(BaseSchema):
    inn = fields.String(validate=InnFieldValidators(), missing=None)
    services = fields.List(fields.Integer(), required=True, allow_none=False)
    require_online = fields.Boolean(required=True, allow_none=False)
    categories = fields.List(fields.Integer(), required=True, allow_none=False)
    spark_id = fields.Integer()
    contact = fields.Nested(PersonSchema)
    functionality = fields.Nested(FunctionalityAPISchema, missing={'type': FunctionalityType.PAYMENTS.value})


class DumpMerchantPreregistrationSchema(BaseSchema):
    preregister_data = fields.Nested(DumpPreregisterDataSchema, dump_only=True)
    raw_preregister_data = fields.Nested(DumpPreregisterDataSchema, dump_only=True)


class GetMerchantPreregistrationResponseSchema(BaseSchema):
    data = fields.Nested(DumpMerchantPreregistrationSchema)


class ServiceSchema(BaseSchema):
    service_id = fields.Integer()
    name = fields.String()


class ServiceMerchantSchema(BaseSchema):
    uid = fields.Integer()
    service_id = fields.Integer()
    entity_id = fields.String()
    description = fields.String()
    service_merchant_id = fields.Integer()
    enabled = fields.Boolean()
    created = fields.DateTime()
    updated = fields.DateTime()
    deleted = fields.Boolean()
    revision = fields.Integer()

    client_id = fields.String()
    person_id = fields.String()
    contract_id = fields.String()
    submerchant_id = fields.String()

    service = fields.Nested(ServiceSchema)

    acquirer = EnumField(AcquirerType, by_value=True, dump_only=True)
    trustworthy = fields.Boolean(dump_only=True)
    status = EnumField(MerchantStatus, by_value=True, dump_only=True)
    organization = fields.Nested(OrganizationSchema, dump_only=True)
    addresses = fields.Nested(AddressesSchema, dump_only=True)
    moderation = fields.Nested(ModerationSchema, dump_only=True)

    @pre_dump
    def pre_dump(self, data):
        result = asdict(data['service_merchant'])

        merchant: Merchant = data.get('merchant')
        if merchant is not None:
            result['status'] = merchant.status
            result['trustworthy'] = merchant.trustworthy
            result['acquirer'] = merchant.acquirer
            result['client_id'] = merchant.client_id
            result['person_id'] = merchant.person_id
            result['contract_id'] = merchant.contract_id
            result['submerchant_id'] = merchant.submerchant_id

            if merchant.status == MerchantStatus.DRAFT:
                merchant_data = create_merchant_draft_schema.load(
                    show_merchant_draft_schema.dump(merchant).data
                ).data
                result['moderation'] = None
                result['organization'] = merchant_data.get('organization')
                result['addresses'] = merchant_data.get('addresses')
            else:
                result['moderation'] = asdict(merchant.moderation) if merchant.moderation else None
                result['organization'] = asdict(merchant.organization) if merchant.organization else None
                result['addresses'] = (
                    [asdict(address) for address in merchant.addresses]
                    if merchant.addresses
                    else None
                )
        return result


class MerchantResponseSchema(SuccessResponseSchema):
    data = fields.Nested(DumpMerchantSchema)


class DeveloperKeySchema(BaseSchema):
    key = fields.String(required=True)


class MerchantTokenSchema(BaseSchema):
    token = fields.String(required=True)


class MerchantTokenResponseSchema(SuccessResponseSchema):
    data = fields.Nested(MerchantTokenSchema)


class MerchantServiceListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(ServiceMerchantSchema, many=True)


class MerchantServiceRequestSchema(BaseSchema):
    enabled = fields.Boolean()
    description = fields.String()


class PointSchema(BaseSchema):
    x = fields.DateTime()
    y = fields.Decimal(places=2)


class AllPaymentsGraphResponseSchema(BaseSchema):
    data = fields.Nested(PointSchema, many=True)


class AllPaymentsGraphRequestInfoSchema(UIDRequestSchema):
    graph_type = EnumField(GraphType, by_value=True)


class AllPaymentsGraphRequestQuerySchema(BaseSchema):
    lower_dt = fields.DateTime(required=True)
    upper_dt = fields.DateTime(required=True)
    group_by = EnumField(GroupType, by_value=True)
    pay_method = fields.String(default=None, validate=validate.OneOf(PAY_METHODS))
    shop_id = fields.Integer()


class OAuthCompleteRequestSchema(BaseSchema):
    code = fields.String(required=True)
    shop_id = fields.Integer()


class OAuthDeleteRequestSchema(BaseSchema):
    merchant_oauth_mode = EnumField(
        MerchantOAuthMode,
        by_value=True,
        missing=MerchantOAuthMode.TEST,
        validate=validate.OneOf([MerchantOAuthMode.TEST])
    )


class OAuthStartRequestSchema(BaseSchema):
    # https://kassa.yandex.ru/developers/partners-api/quick-start#requesting-token-redirect
    # Строка состояния, которую Яндекс.OAuth возвращает без изменения.
    # Возможно нужна фронту при редиректе обратно из Кассы.
    state = fields.String(validate=validate.Length(max=1024))


class OAuthStartUrlSchema(BaseSchema):
    url = fields.String(required=True)


class OAuthStartResponseSchema(SuccessResponseSchema):
    data = fields.Nested(OAuthStartUrlSchema)


class MerchantSuggestRequestSchema(BaseSchema):
    query = StripString(required=True)

    @pre_load
    def clean_punctuation(self, data: dict) -> dict:
        if 'query' in data:
            regex = re.compile(f'[{re.escape(string.punctuation)} ]+')
            data['query'] = regex.sub(' ', data['query']).strip()
        return data


class MerchantSuggestItemSchema(BaseSchema):
    spark_id = fields.Integer()
    name = fields.String()
    full_name = fields.String()
    inn = fields.String()
    ogrn = fields.String()
    address = fields.String()
    leader_name = fields.String()
    region_name = fields.String()


class MerchantSuggestResponseSchema(SuccessResponseSchema):
    data = fields.Nested(MerchantSuggestItemSchema, many=True)


class MerchantOrdersStatsSchema(BaseSchema):
    orders_sum = fields.Decimal(places=2)
    orders_paid_count = fields.Integer()
    orders_created_count = fields.Integer()
    money_average = fields.Decimal(places=2)


class MerchantOrdersStatsResponseSchema(SuccessResponseSchema):
    data = fields.Nested(MerchantOrdersStatsSchema)


class MerchantOrdersStatsRequestSchema(BaseSchema):
    date_from = fields.DateTime()
    date_to = fields.DateTime()


create_merchant_draft_schema = CreateMerchantDraftSchema()
show_merchant_draft_schema = DumpMerchantSchema()
developer_key_schema = DeveloperKeySchema()

merchant_request_schema = CreateMerchantSchema()
merchant_response_schema = MerchantResponseSchema()

merchant_draft_request_schema = CreateMerchantDraftSchema()

merchant_preregister_request_schema = CreateMerchantPreregisterSchema()
get_merchant_preregistration_response_schema = GetMerchantPreregistrationResponseSchema()

merchant_token_response_schema = MerchantTokenResponseSchema()

merchant_service_request_schema = MerchantServiceRequestSchema()

merchant_service_list_response_schema = MerchantServiceListResponseSchema()

payments_graph_request_info_schema = AllPaymentsGraphRequestInfoSchema()
payments_graph_request_query_schema = AllPaymentsGraphRequestQuerySchema()
payments_graph_response_schema = AllPaymentsGraphResponseSchema()

oauth_start_request_schema = OAuthStartRequestSchema()
oauth_start_response_schema = OAuthStartResponseSchema()

oauth_complete_request_schema = OAuthCompleteRequestSchema()

oauth_delete_request_schema = OAuthDeleteRequestSchema()

merchant_suggest_request_schema = MerchantSuggestRequestSchema()
merchant_suggest_response_schema = MerchantSuggestResponseSchema()

merchant_orders_stats_request_schema = MerchantOrdersStatsRequestSchema()
merchant_orders_stats_response_schema = MerchantOrdersStatsResponseSchema()
