from marshmallow import ValidationError, fields, post_dump, post_load, pre_dump, pre_load, validate, validates_schema
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import (
    CURRENCY_RUB, BasePaginatedRequestSchema, BaseSchema, SuccessResponseSchema
)
from mail.payments.payments.api.schemas.customer_subscription import CustomerSubscription
from mail.payments.payments.api.schemas.fields import NullableBoolean, QueryParamList, StripEmail, StripString
from mail.payments.payments.api.schemas.mixins import PayTokenMixin
from mail.payments.payments.api.schemas.path import UIDServiceMerchantIDRequestSchema
from mail.payments.payments.api.schemas.shop import ShopSchema
from mail.payments.payments.core.entities.enums import (
    NDS, PAY_METHOD_OFFLINE, PAY_METHOD_YANDEX, PAY_METHODS, PAYMETHOD_ID_OFFLINE, AcquirerType, OrderKind, OrderSource,
    OrderTimelineEventType, PayStatus, ReceiptType, RefundStatus, ShopType
)


class OrderItemImageStoredSchema(BaseSchema):
    original = fields.String(attribute='orig')


class OrderItemImageSchema(BaseSchema):
    stored = fields.Nested(OrderItemImageStoredSchema, dump_only=True)
    url = fields.String(required=True)


class OrderItemSchema(BaseSchema):
    amount = fields.Decimal(places=2, required=True)
    currency = fields.String(validate=validate.Equal(CURRENCY_RUB), required=True)
    name = StripString(required=True, validate=validate.Length(max=128))
    nds = EnumField(NDS, by_value=True, required=True)
    price = fields.Decimal(places=2, required=True)
    image = fields.Nested(OrderItemImageSchema)
    markup = fields.Dict(allow_none=True)

    product_id = fields.Integer(dump_only=True)

    @pre_load
    def default_total_price(self, data):
        if 'total_price' in data:
            data.setdefault('price', data['total_price'])
        return data

    @validates_schema
    def validate_item(self, data):
        price = data.get('price')
        amount = data.get('amount')
        if price is not None and price <= 0 or amount is not None and amount <= 0:
            raise ValidationError('Order item must have positive price and amount')


class EmailContextSchema(BaseSchema):
    to_email = StripEmail(required=True)
    reply_email = StripEmail(required=True)


class OrderServiceSchema(BaseSchema):
    service_id = fields.Integer()
    name = fields.String()


class OrderServiceMechantSchema(BaseSchema):
    enabled = fields.Boolean()
    entity_id = fields.String()
    deleted = fields.Boolean()
    created = fields.DateTime()
    service = fields.Nested(OrderServiceSchema)


class OriginalOrderInfoSchema(BaseSchema):
    pay_method = fields.String()
    paymethod_id = fields.String()
    pay_status = EnumField(PayStatus, by_value=True)


class TimelineEventPeriodicTxId(BaseSchema):
    uid = fields.Integer(required=True)
    customer_subscription_id = fields.Integer(required=True)
    purchase_token = fields.String(required=True)


class TimelineEventExtraSchema(BaseSchema):
    refund_amount = fields.Decimal(places=2)
    periodic_amount = fields.Decimal(places=2)
    tx_id = fields.Nested(TimelineEventPeriodicTxId, dump_only=True)


class OrderTimelineEventSchema(BaseSchema):
    date = fields.DateTime()
    event_type = EnumField(OrderTimelineEventType, by_value=True)
    extra = fields.Nested(TimelineEventExtraSchema)


class OrderSchemaBase(BaseSchema):
    active = fields.Boolean()
    caption = StripString()
    held_at = fields.DateTime()
    description = fields.String()
    kind = EnumField(OrderKind, by_value=True)
    order_id = fields.Integer()
    original_order_id = fields.Integer()
    revision = fields.Integer()
    uid = fields.Integer()
    paymethod_id = fields.String()
    customer_uid = fields.Integer()
    verified = fields.Boolean()
    service_merchant_id = fields.Integer()

    offline_abandon_deadline = fields.DateTime()
    created = fields.DateTime()
    updated = fields.DateTime()
    pay_status_updated_at = fields.DateTime()

    pay_status = EnumField(PayStatus, by_value=True)
    refund_status = EnumField(RefundStatus, by_value=True)

    trust_resp_code = fields.String(dump_only=True)

    price = fields.Decimal(places=2, dump_only=True)
    currency = fields.String(validate=validate.Equal(CURRENCY_RUB), dump_only=True)
    items = fields.Nested(OrderItemSchema, many=True, dump_only=True)
    service_merchant = fields.Nested(OrderServiceMechantSchema, dump_only=True)
    shop = fields.Nested(ShopSchema, dump_only=True)
    original_order_info = fields.Nested(OriginalOrderInfoSchema, dump_only=True)

    email = fields.Nested(EmailContextSchema, dump_only=True, attribute='email_context')

    shop_type = EnumField(ShopType, attribute='shop.shop_type', dump_only=True, dump_to='mode', by_value=True)
    customer_subscription = fields.Nested(CustomerSubscription, dump_only=True)
    timeline = fields.Nested(OrderTimelineEventSchema, dump_only=True, many=True)
    fast_moderation = fields.Boolean(dump_only=True)

    # For multi
    max_amount = fields.Integer(dump_only=True, attribute='data.multi_max_amount')
    issued_amount = fields.Integer(dump_only=True, attribute='data.multi_issued')

    @pre_dump
    def fix_items_pre(self, data):
        if isinstance(data, dict):
            data.setdefault('items', None)
        return data

    @post_dump
    def fix_items_post(self, data):
        if not data.get('items'):
            data.pop('items', None)
        return data

    @post_dump
    def refund_order_fix(self, data):
        if data.get('kind') != OrderKind.MULTI.value:
            data.pop('max_amount', None)
            data.pop('issued_amount', None)

        if data.get('kind') != OrderKind.PAY.value:
            data.pop('autoclear', None)
            data.pop('pay_status', None)

        if data.get('kind') != OrderKind.REFUND.value:
            data.pop('original_order_id', None)
            data.pop('refund_status', None)

        return data


class RefundSchema(OrderSchemaBase):
    class Meta:
        strict = True
        exclude = (
            'items.image',
        )


class OrderSchema(OrderSchemaBase):
    autoclear = fields.Boolean()
    closed = fields.DateTime()
    user_description = fields.String()
    user_email = StripString()
    order_url = fields.String(dump_only=True)
    order_hash = fields.String(dump_only=True)
    payment_hash = fields.String(dump_only=True)
    payment_url = fields.String(dump_only=True)
    pay_method = fields.String()
    refunds = fields.Nested(RefundSchema, many=True, dump_only=True)


class OrderResponseSchema(SuccessResponseSchema):
    data = fields.Nested(OrderSchema)


class GetOrderListRequestSchema(BasePaginatedRequestSchema):
    created_from = fields.DateTime(load_from='lower_dt')
    created_to = fields.DateTime(load_from='upper_dt')
    held_at_from = fields.DateTime(load_from='lower_held_dt')
    held_at_to = fields.DateTime(load_from='upper_held_dt')
    price_from = fields.Decimal(load_from='lower_price')
    price_to = fields.Decimal(load_from='upper_price')
    kinds = QueryParamList(EnumField(OrderKind, by_value=True))
    pay_statuses = QueryParamList(EnumField(PayStatus, by_value=True))
    refund_statuses = QueryParamList(EnumField(RefundStatus, by_value=True))
    pay_method = fields.String(default=None, validate=validate.OneOf(PAY_METHODS))
    is_active = fields.Boolean()
    text_query = fields.String()
    email_query = fields.String()
    sort_by = fields.String(validate=validate.OneOf(['created', 'updated', 'held_at', 'price']))
    parent_order_id = fields.Integer()
    original_order_id = fields.Integer()
    descending = fields.Boolean(load_from='desc')
    created_by_sources = QueryParamList(EnumField(OrderSource, by_value=True))
    service_ids = QueryParamList(fields.Integer(strict=True))
    subscription = NullableBoolean(missing=False)
    with_refunds = fields.Boolean()


class OrderListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(OrderSchema, many=True)


class PostOrderRequestSchema(BaseSchema):
    kind = EnumField(OrderKind, by_value=True, missing=lambda: OrderKind.PAY)
    autoclear = fields.Boolean(missing=True)
    caption = StripString(required=False)
    description = fields.String()
    return_url = fields.String()
    items = fields.Nested(OrderItemSchema, many=True)
    max_amount = fields.Integer(validate=validate.Range(min=1), missing=None)
    offline_abandon_deadline = fields.DateTime()
    default_shop_type = EnumField(ShopType, by_value=True, load_only=True, load_from='mode')

    paymethod_id = fields.String(validate=validate.OneOf([None]))
    pay_method = fields.String(default=None, validate=validate.OneOf([PAY_METHOD_YANDEX, PAY_METHOD_OFFLINE]))

    fast_moderation = fields.Boolean(missing=False)

    @post_load
    def post_load(self, data):
        # Hack for PAYBACK-400
        if data.get('kind') not in (None, OrderKind.PAY, OrderKind.MULTI):
            raise ValidationError(
                f'kind must be one of ("{OrderKind.PAY.value}", "{OrderKind.MULTI.value}")', field_names=['kind']
            )

        if not data.get('items'):
            raise ValidationError('Must not be empty.', field_names=['items'])

        if data.get('kind') != OrderKind.MULTI:
            if data.get('max_amount'):
                raise ValidationError(f'kind should be {OrderKind.MULTI.value}', field_names=['max_amount'])
            else:
                data.pop('max_amount', None)

        # PAYBACK-550
        pay_method = data.pop('pay_method', None)
        if pay_method:
            if 'paymethod_id' in data:
                raise ValidationError(
                    'pay_method should be passed without paymethod_id', ['paymethod_id', 'pay_method']
                )
            elif pay_method == PAY_METHOD_OFFLINE:
                data['paymethod_id'] = PAYMETHOD_ID_OFFLINE
            else:
                data['paymethod_id'] = None

        return data


class PutOrderRequestSchema(PostOrderRequestSchema):
    class Meta:
        strict = True
        fields = ('caption', 'description', 'items', 'autoclear', 'offline_abandon_deadline', 'paymethod_id',
                  'fast_moderation')


class ActiveOrderRequestSchema(BaseSchema):
    active = fields.Boolean()


class PostOrderClearUnholdPathSchema(UIDServiceMerchantIDRequestSchema):
    order_id = fields.Integer(required=True)
    operation = fields.String(required=True, validate=validate.OneOf(['clear', 'unhold']))


class PostRefundRequestSchema(PostOrderRequestSchema):
    class Meta:
        strict = True
        fields = ('caption', 'description', 'items')


class PostRefundCustomerSubscriptionTransactionRequestSchema(BaseSchema):
    caption = StripString(required=False)
    description = fields.String()


class PostOrderEmailSchema(EmailContextSchema):
    spam_check = fields.Boolean(required=False)


class PostOrderEmailResponseSchema(SuccessResponseSchema):
    data = fields.Nested(EmailContextSchema)


class PostRefundResponseSchema(SuccessResponseSchema):
    data = fields.Nested(RefundSchema)


class PayOfflineOrderRequestSchema(BaseSchema):
    customer_uid = fields.Integer(missing=None, allow_none=True)
    description = fields.String()


class OrderParamsRequestSchema(BaseSchema):
    with_timeline = fields.Boolean()


class PayTokenOrderSchema(PayTokenMixin, OrderSchema):
    class Meta:
        strict = True
        exclude = (
            'order_hash',
            'order_url',
            'payment_hash',
            'payment_url',
        )

    payments_url = fields.String(dump_only=True)
    trust_url = fields.String(dump_only=True)
    service_data = fields.Dict(allow_none=True)
    refunds = fields.Nested(RefundSchema, many=True)
    acquirer = EnumField(AcquirerType, by_value=True, dump_only=True)
    receipt_type = EnumField(ReceiptType, by_value=True, attribute='data.receipt_type')


get_order_response_schema = OrderResponseSchema()

get_order_list_request_schema = GetOrderListRequestSchema()
get_order_list_response_schema = OrderListResponseSchema()

post_order_request_schema = PostOrderRequestSchema()
post_order_response_schema = OrderResponseSchema()

pay_offline_order_response_schema = OrderResponseSchema()

post_order_email_schema = PostOrderEmailSchema()
post_order_email_response_schema = PostOrderEmailResponseSchema()

put_order_request_schema = PutOrderRequestSchema()
put_order_response_schema = OrderResponseSchema()

active_order_request_schema = ActiveOrderRequestSchema()
active_order_response_schema = OrderResponseSchema()

post_order_v1_clear_unhold_path_schema = PostOrderClearUnholdPathSchema(exclude=('service_merchant_id',))
post_order_internal_clear_unhold_path_schema = PostOrderClearUnholdPathSchema(exclude=('uid',))

post_refund_request_schema = PostRefundRequestSchema()
post_refund_response_schema = PostRefundResponseSchema()

post_refund_customer_subscription_tx_schema = PostRefundCustomerSubscriptionTransactionRequestSchema()

post_pay_offline_order_request_schema = PayOfflineOrderRequestSchema()

order_params_request_schema = OrderParamsRequestSchema()
