from marshmallow import ValidationError, fields, post_load, pre_dump
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.fields import StripEmail
from mail.payments.payments.api.schemas.merchant import AddressSchema
from mail.payments.payments.conf import settings
from mail.payments.payments.core.entities.enums import AcquirerType, TrustEnv
from mail.payments.payments.core.entities.merchant import Merchant


class PaymentSystemsOptionsSchema(BaseSchema):
    apple_pay_enabled = fields.Boolean()
    google_pay_enabled = fields.Boolean()


class MerchantSchema(BaseSchema):
    name = fields.String()
    ogrn = fields.String()
    schedule_text = fields.String()
    legal_address = fields.Nested(AddressSchema)

    @pre_dump
    def dump_merchant(self, m: Merchant) -> dict:
        return dict(
            name=m.name,
            ogrn=m.organization.ogrn,
            schedule_text=m.organization.schedule_text,
            legal_address=m.get_address_by_type('legal'),
        )


class PurchaseTokenSchema(BaseSchema):
    purchase_token = fields.String(required=True)
    acquirer = EnumField(AcquirerType, by_value=True, dump_only=True)
    environment = EnumField(TrustEnv, by_value=True, dump_only=True)
    payment_systems_options = fields.Nested(PaymentSystemsOptionsSchema)
    merchant = fields.Nested(MerchantSchema, dump_only=True)


class PostOrderStartSchema(BaseSchema):
    pay_token = fields.String(required=True)
    email = StripEmail(allow_none=True)
    customer_uid = fields.Integer()
    turboapp_id = fields.String(allow_none=True)
    tsid = fields.String(allow_none=True)
    psuid = fields.String(allow_none=True)

    @post_load
    def post_load(self, data):
        if not data.get('pay_token').startswith(settings.PAY_TOKEN_PREFIX):
            raise ValidationError('Invalid format.', field_names=['pay_token'])


class PostOrderStartResponseSchema(SuccessResponseSchema):
    data = fields.Nested(PurchaseTokenSchema)


post_order_start_schema = PostOrderStartSchema()
post_order_start_response_schema = PostOrderStartResponseSchema()
