from typing import Any, Dict

from marshmallow import fields, post_dump
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BasePaginatedRequestSchema, BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.merchant import OAuthSchema
from mail.payments.payments.core.entities.enums import ShopType


class ShopSchema(BaseSchema):
    uid = fields.Integer()
    shop_id = fields.Integer()
    name = fields.String()
    is_default = fields.Boolean()
    shop_type = EnumField(ShopType, by_value=True)
    created = fields.DateTime()
    updated = fields.DateTime()
    oauth = fields.Nested(OAuthSchema, default=None)

    @post_dump
    def null_empty_oauth(self, data: Dict[str, Any]) -> dict:
        if not data.get('oauth'):
            data['oauth'] = None
        return data


class GetShopListRequestSchema(BasePaginatedRequestSchema):
    pass


class GetShopListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(ShopSchema, many=True)


class PostShopListRequestSchema(BaseSchema):
    name = fields.String(required=True)
    is_default = fields.Boolean(default=False)
    shop_type = EnumField(ShopType, by_value=True, required=True)


class PostShopListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(ShopSchema)


class PutShopRequestSchema(BaseSchema):
    name = fields.String(required=True)
    is_default = fields.Boolean(required=True)


class PutShopResponseSchema(SuccessResponseSchema):
    data = fields.Nested(ShopSchema)


get_shop_list_request_schema = GetShopListRequestSchema()
get_shop_list_response_schema = GetShopListResponseSchema()

post_shop_list_request_schema = PostShopListRequestSchema()
post_shop_list_response_schema = PostShopListResponseSchema()

put_shop_request_schema = PutShopRequestSchema()
put_shop_response_schema = PutShopResponseSchema()
