from marshmallow import ValidationError, fields, validates_schema
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BaseSchema
from mail.payments.payments.api.schemas.merchant import DumpMerchantSchema
from mail.payments.payments.core.entities.enums import MerchantRole


class UserRoleRequestSchema(BaseSchema):
    user_uid = fields.Integer()
    user_email = fields.Email()
    role = EnumField(MerchantRole, by_value=True, required=True)
    description = fields.String(allow_none=True)

    @validates_schema
    def validate_role_assignment(self, data):
        user_uid_present = data.get('user_uid') is not None
        user_email_present = data.get('user_email') is not None
        if user_uid_present == user_email_present:
            raise ValidationError('Exactly one of "user_email", "user_uid" is required')


class UserRoleSchema(BaseSchema):
    uid = fields.Integer(dump_to='user_uid')
    email = fields.String(dump_to='user_email')
    merchant_id = fields.String()
    role = EnumField(MerchantRole, by_value=True)
    description = fields.String()
    updated = fields.DateTime(dump_only=True)
    created = fields.DateTime(dump_only=True)


class UserRoleWithMerchantSchema(UserRoleSchema):
    merchant = fields.Nested(DumpMerchantSchema())


class UserRoleResponseSchema(BaseSchema):
    data = fields.Nested(UserRoleSchema)


class UserRoleListRequestQuerySchema(BaseSchema):
    role = EnumField(MerchantRole, by_value=True)


class MerchantUserRoleListResponseSchema(BaseSchema):
    data = fields.Nested(UserRoleSchema, many=True)


class UserRoleListResponseSchema(BaseSchema):
    data = fields.Nested(UserRoleWithMerchantSchema, many=True)


class DeleteUserRoleRequestSchema(BaseSchema):
    user_uid = fields.Integer()
    user_email = fields.Email()

    @validates_schema
    def validate_role_deletion(self, data):
        if data.get('user_uid') is None and data.get('user_email') is None:
            raise ValidationError('Either user_email or user_uid required')


user_role_request_schema = UserRoleRequestSchema()
user_role_response_schema = UserRoleResponseSchema()

user_role_list_request_query_schema = UserRoleListRequestQuerySchema()
user_role_list_response_schema = UserRoleListResponseSchema()
merchant_user_role_list_response_schema = MerchantUserRoleListResponseSchema()

delete_user_role_request_schema = DeleteUserRoleRequestSchema()
