from typing import Callable, Iterable, Optional

from aiohttp.web_middlewares import normalize_path_middleware

from sendr_aiohttp import Url, create_apispec, setup_swagger_route
from sendr_aiopg.engine.single import CustomEngine

from mail.payments.payments import __version__
from mail.payments.payments.api.app import DBEngineApplication
from mail.payments.payments.api.middlewares import (
    middleware_header_cloud, middleware_logging_adapter, middleware_response_formatter
)
from mail.payments.payments.api.routes.base import UrlDispatcher
from mail.payments.payments.api_admin.middlewares import middleware_tvm
from mail.payments.payments.api_admin.routes import idm, ping, v1, v2
from mail.payments.payments.conf import settings
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.storage.writers import PaymentsPushers


class AdminApplication(DBEngineApplication):
    _urls: Iterable[Url] = (
        *idm.URLS,
        *ping.URLS,
        *v1.URLS,
        *v2.URLS,
    )
    _middlewares: Iterable[Callable] = (
        # ...,  # todo: logging middleware for SOC
        normalize_path_middleware(remove_slash=False, append_slash=True),
        middleware_header_cloud,
        middleware_logging_adapter,
        middleware_response_formatter,
        middleware_tvm,
    )

    router: UrlDispatcher

    def __init__(self, db_engine: CustomEngine, crypto: Optional[Crypto] = None):
        super().__init__(
            db_engine=db_engine,
            router=UrlDispatcher(),
            middlewares=self._middlewares
        )

        self.pushers = PaymentsPushers()
        self.router.add_routes(self._urls)
        self.add_sentry()
        self.crypto = crypto or Crypto.from_file(settings.CRYPTO_KEYS_FILE)

        if settings.SWAGGER_ENABLED:
            spec = create_apispec(
                app=self,
                title='Payments Managing Panel API',
                version=__version__,
            )
            setup_swagger_route(
                app=self,
                spec=spec,
                swagger_route='/admin/api/swagger.json',
                swagger_route_name='swagger',
            )

        self.on_cleanup.append(self.pushers.close)

    def add_sentry(self):
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            self.on_cleanup.append(sentry_init(settings.SENTRY_DSN, release=__version__))
