from typing import Optional

from aiohttp.abc import StreamResponse

from sendr_tvm.qloud_async_tvm import TicketCheckResult

from mail.payments.payments.api.exceptions import TVMServiceTicketException, TVMUserTicketException
from mail.payments.payments.api.handlers.base import BaseHandler
from mail.payments.payments.conf import settings


class BaseAdminHandler(BaseHandler):
    """
    Admin endpoints must receive user TVM ticket
    Assuming middleware inject TVM ticket check result into request['tvm'] entry,
    this base handler checks incoming request to have user TVM ticket in headers
    """
    _uid: Optional[int] = None

    async def get_data(self) -> dict:
        data = await super().get_data()
        return {
            **data,
            'manager_uid': self.uid,
        }

    @property
    def uid(self) -> int:
        """UID from user ticket"""
        assert self._uid is not None
        return self._uid

    def verify_tvm(self) -> bool:
        """Assuming there is TVM middleware, which injects check result of TVM ticket into request object"""
        if not settings.TVM_CHECK_SERVICE_TICKET:
            return True
        check_result: TicketCheckResult = self.tvm
        return check_result.src in settings.TVM_ADMIN_API_ALLOWED_CLIENTS

    def _get_uid_from_user_ticket(self) -> int:
        if settings.DEBUG:
            try:
                return int(self.request.headers.get('X-Ya-Uid-Debug'))
            except (ValueError, TypeError):
                pass

        if self.tvm.default_uid is None:
            raise TVMUserTicketException(message='Missing User-Ticket')
        try:
            uid = int(self.tvm.default_uid)
        except (TypeError, ValueError):
            raise TVMUserTicketException(message='Unable to determine UID from User-Ticket')
        return uid

    async def _iter(self) -> StreamResponse:
        if not self.verify_tvm():  # idm is not allowed
            raise TVMServiceTicketException
        self._uid = self._get_uid_from_user_ticket()
        return await super()._iter()
