from asyncio import CancelledError
from typing import Any, Dict

from aiohttp import web
from aiohttp.abc import StreamResponse

from sendr_aiohttp.handler import request_schema
from sendr_tvm.qloud_async_tvm import TicketCheckResult

from mail.payments.payments.api.handlers.base import BaseHandler
from mail.payments.payments.api_admin.schemas.idm import idm_request_schema
from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.manager.role import (
    AddRoleManagerAction, ListAllManagerRolesAction, RemoveRoleManagerAction
)
from mail.payments.payments.core.entities.enums import Role
from mail.payments.payments.core.exceptions import BaseCoreError

ROLES = {
    'slug': 'role',
    'name': {
        'ru': 'роль',
        'en': 'role'
    },
    'values': {
        Role.ADMIN.value: {
            'name': {'ru': 'Администратор', 'en': 'Admin'},
        },
        Role.ASSESSOR.value: {
            'name': {'ru': 'Ассесор', 'en': 'Assessor'},
        },
        Role.ACCOUNTANT.value: {
            'name': {'ru': 'Бухгалтер', 'en': 'Accountant'},
        },
    }
}


class IDMBaseHandler(BaseHandler):
    class BaseHandlerError(Exception):
        def __init__(self, *args: Any, message: str = ''):
            self.message = message
            super().__init__(*args)

    class Warning(BaseHandlerError):
        pass

    class Error(BaseHandlerError):
        pass

    class Fatal(BaseHandlerError):
        pass

    async def success_response(self, data: Dict[str, Any]) -> web.Response:
        """IDM-compatible success response"""
        data['code'] = 0
        return self.make_response(data)

    async def error_response(self, message: str) -> web.Response:
        """IDM treats this as kind of success response but will log it"""
        return self.make_response({
            'code': 1,
            'error': message,
        })

    async def warning_response(self, message: str) -> web.Response:
        """IDM will retry request after some time when receiving this kind of response"""
        return self.make_response({
            'code': 1,
            'warning': message,
        })

    async def fatal_response(self, message: str) -> web.Response:
        """IDM will not retry request after this response"""
        return self.make_response({
            'code': 1,
            'fatal': message,
        })

    async def verify_idm_request(self) -> bool:
        """Assuming there is TVM middleware, which injects check result of TVM ticket into request object"""
        if not settings.TVM_CHECK_IDM_CLIENT_ID:
            return True
        check_result: TicketCheckResult = self.tvm
        try:
            return int(check_result.src) == int(settings.TVM_IDM_CLIENT_ID)
        except (TypeError, ValueError):
            self.logger.warning("Unable to verify IDM TVM ID")
            return False

    async def _iter(self) -> StreamResponse:
        """Forces IDM-compatible responses - always 200 OK and error is encoded in response body"""
        if not await self.verify_idm_request():  # important
            return await self.error_response('Request source not allowed')
        try:
            return await super()._iter()
        # one may explicitly return from handler or raise specific exception
        except self.Warning as e:
            return await self.warning_response(e.message)
        except self.Error as e:
            return await self.error_response(e.message)
        except self.Fatal as e:
            return await self.fatal_response(e.message)
        except CancelledError:
            raise
        except Exception:
            return await self.error_response('Unexpected error')

    def _core_exception_result(self, exc: BaseCoreError) -> None:
        # do not want to raise APIException instead of exception from Action
        raise exc


class IDMInfoHandler(IDMBaseHandler):
    async def get(self):
        return await self.success_response({'roles': ROLES})


class IDMAddRoleHandler(IDMBaseHandler):
    @request_schema(idm_request_schema, location='form')
    async def post(self):
        await self.run_action(AddRoleManagerAction, await self.get_data())
        return await self.success_response({})


class IDMRemoveRoleHandler(IDMBaseHandler):
    @request_schema(idm_request_schema, location='form')
    async def post(self):
        await self.run_action(RemoveRoleManagerAction, await self.get_data())
        return await self.success_response({})


class IDMGetAllRolesHandler(IDMBaseHandler):
    async def get(self):
        role_descriptions = await self.run_action(ListAllManagerRolesAction)
        return await self.success_response({'users': role_descriptions})
