from aiohttp import web

from sendr_aiohttp.handler import request_schema, response_schema

from mail.payments.payments.api.schemas.base import success_response_schema
from mail.payments.payments.api.schemas.path import uid_request_schema
from mail.payments.payments.api_admin.handlers.base import BaseAdminHandler
from mail.payments.payments.api_admin.schemas.merchant import (
    block_merchant_request_schema, get_merchant_list_request_schema, get_merchant_list_request_schema_v2,
    get_merchant_list_response_schema, get_merchant_list_response_schema_v2, support_comment_request_schema
)
from mail.payments.payments.core.actions.manager.merchant import (
    BlockMerchantManagerAction, GetMerchantListManagerAction, RecreateMerchantModerationManagerAction,
    UpdateSupportCommentManagerAction
)


class MerchantHandler(BaseAdminHandler):
    @request_schema(get_merchant_list_request_schema, location='query')
    @response_schema(get_merchant_list_response_schema)
    async def get(self) -> web.Response:
        merchants_data = await self.run_action(GetMerchantListManagerAction, await self.get_data())
        return self.make_response({'data': merchants_data.merchants}, get_merchant_list_response_schema)


class MerchantHandlerV2(BaseAdminHandler):
    @request_schema(get_merchant_list_request_schema_v2, location='query')
    @response_schema(get_merchant_list_response_schema_v2)
    async def get(self) -> web.Response:
        merchants_data = await self.run_action(GetMerchantListManagerAction, await self.get_data())

        response_data = {
            'data': {
                'merchants': merchants_data.merchants,
                'total': merchants_data.stats.total,
                'found': merchants_data.stats.found,
            }
        }
        if merchants_data.keyset is not None:
            response_data['data']['next'] = {'keyset': merchants_data.keyset}

        return self.make_response(
            response_data,
            get_merchant_list_response_schema_v2,
        )


class SupportCommentMerchantHandler(BaseAdminHandler):
    @request_schema(uid_request_schema, location='match_info')
    @request_schema(support_comment_request_schema, location='json')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(UpdateSupportCommentManagerAction, await self.get_data())
        return self.make_response({}, success_response_schema)


class BlockMerchantHandler(BaseAdminHandler):
    @request_schema(uid_request_schema, location='match_info')
    @request_schema(block_merchant_request_schema, location='json')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(BlockMerchantManagerAction, await self.get_data())
        return self.make_response({}, success_response_schema)


class RecreateMerchantModerationHandler(BaseAdminHandler):
    @request_schema(uid_request_schema, location='match_info')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(RecreateMerchantModerationManagerAction, await self.get_data())
        return self.make_response({}, success_response_schema)
