from marshmallow import fields, validate
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.fields import QueryParamList
from mail.payments.payments.api.schemas.keyset import (
    BaseKeysetEntitySchema, KeysetEntryDateTimeSchema, KeysetEntryDecimalSchema, KeysetEntryIntegerSchema
)
from mail.payments.payments.api.schemas.merchant import OrganizationSchema, ServiceSchema
from mail.payments.payments.api_admin.schemas.moderation import AdminModerationSchema
from mail.payments.payments.core.entities.enums import AcquirerType, ModerationStatus
from mail.payments.payments.core.entities.keyset import ManagerAnalyticsKeysetEntity


class AnalyticsKeysetEntitySchema(BaseKeysetEntitySchema):
    KEYSET_ENTITY_CLS = ManagerAnalyticsKeysetEntity

    uid = fields.Nested(KeysetEntryIntegerSchema)
    created = fields.Nested(KeysetEntryDateTimeSchema)
    payments_success = fields.Nested(KeysetEntryDecimalSchema)
    money_success = fields.Nested(KeysetEntryDecimalSchema)


class GetMerchantAnalyticsRequestSchema(BaseSchema):
    limit = fields.Integer(missing=10, validate=validate.Range(max=50))

    # filters
    merchant_uid = fields.Integer()
    name = fields.String()
    created_from = fields.DateTime()
    created_to = fields.DateTime()
    order_created_from = fields.DateTime()
    order_created_to = fields.DateTime()
    order_closed_from = fields.DateTime()
    order_closed_to = fields.DateTime()
    moderation_status = EnumField(ModerationStatus, by_value=True)
    blocked = fields.Boolean()
    site_url = fields.String()
    acquirer = EnumField(AcquirerType, by_value=True)
    pay_created_from = fields.DateTime()
    pay_created_to = fields.DateTime()
    pay_closed_from = fields.DateTime()
    pay_closed_to = fields.DateTime()
    service_ids = QueryParamList(fields.Integer())

    # sort
    sort_by = fields.String(
        validate=validate.OneOf(['uid', 'created', 'payments_success', 'money_success']),
    )
    desc = fields.Boolean(default=True)
    keyset = fields.Nested(AnalyticsKeysetEntitySchema)


class NextPageSchema(BaseSchema):
    keyset = fields.Nested(AnalyticsKeysetEntitySchema)


class ContactSchema(BaseSchema):
    email = fields.String()


class AdminServiceMerchantSchema(BaseSchema):
    service_merchant_id = fields.Integer()
    service = fields.Nested(ServiceSchema)


class AdminMerchantAnalyticsSchema(BaseSchema):
    uid = fields.Integer()
    name = fields.String()
    moderation = fields.Nested(AdminModerationSchema, only=('updated', 'status'))
    created = fields.DateTime()
    acquirer = EnumField(AcquirerType, by_value=True)
    blocked = fields.Boolean()
    client_id = fields.String()
    organization = fields.Nested(OrganizationSchema, only=('site_url', 'description',))
    service_merchants = fields.Nested(AdminServiceMerchantSchema, many=True)

    payments_total = fields.Integer()
    payments_success = fields.Integer()
    payments_refund = fields.Integer()
    money_aov = fields.Decimal(as_string=True)
    money_success = fields.Decimal(as_string=True)
    money_refund = fields.Decimal(as_string=True)

    support_comment = fields.String()
    contact = fields.Nested(ContactSchema)


class GetMerchantAnalyticsResponseDataSchema(BaseSchema):
    merchants = fields.Nested(AdminMerchantAnalyticsSchema, many=True)
    next_page = fields.Nested(NextPageSchema, dump_to='next', attribute='next')
    total = fields.Integer()
    found = fields.Integer()


class GetMerchantAnalyticsResponseSchema(SuccessResponseSchema):
    data = fields.Nested(GetMerchantAnalyticsResponseDataSchema)


get_merchant_analytics_request_schema = GetMerchantAnalyticsRequestSchema()
get_merchant_analytics_response_schema = GetMerchantAnalyticsResponseSchema()
