from marshmallow import fields, validate
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BasePaginatedRequestSchema, BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.document import DocumentImageSchema
from mail.payments.payments.api.schemas.fields import NestedWithKey, QueryParamList
from mail.payments.payments.api.schemas.keyset import (
    BaseKeysetEntitySchema, KeysetEntryDateTimeSchema, KeysetEntryIntegerSchema
)
from mail.payments.payments.api.schemas.merchant import DumpMerchantSchema
from mail.payments.payments.api_admin.schemas.moderation import AdminModerationSchema
from mail.payments.payments.core.entities.enums import AcquirerType, ModerationStatus
from mail.payments.payments.core.entities.keyset import ManagerMerchantListKeysetEntity


class GetMerchantListRequestSchema(BasePaginatedRequestSchema):
    limit = fields.Integer(missing=10, validate=validate.Range(max=50))

    sort_by = fields.String(
        validate=validate.OneOf(['created', 'updated', 'uid']),
    )
    desc = fields.Boolean(default=True)

    name = fields.String()
    username = fields.String()
    merchant_uid = fields.Integer()
    client_id = fields.String()
    submerchant_id = fields.String()


class GetMerchantListKeysetEntitySchema(BaseKeysetEntitySchema):
    KEYSET_ENTITY_CLS = ManagerMerchantListKeysetEntity

    created = fields.Nested(KeysetEntryDateTimeSchema)
    updated = fields.Nested(KeysetEntryDateTimeSchema)
    uid = fields.Nested(KeysetEntryIntegerSchema)


class GetMerchantListRequestV2Schema(BaseSchema):
    limit = fields.Integer(missing=10, validate=validate.Range(max=50))
    keyset = fields.Nested(GetMerchantListKeysetEntitySchema)

    sort_by = fields.String(
        validate=validate.OneOf(['created', 'updated', 'uid']),
    )
    desc = fields.Boolean(default=True)

    name = fields.String()
    username = fields.String()
    merchant_uid = fields.Integer()
    client_id = fields.String()
    submerchant_id = fields.String()
    acquirers = QueryParamList(EnumField(AcquirerType, by_value=True))
    created_from = fields.DateTime()
    created_to = fields.DateTime()
    updated_from = fields.DateTime()
    updated_to = fields.DateTime()
    moderation_status = EnumField(ModerationStatus, by_value=True)


class AdminMerchantSchema(DumpMerchantSchema):
    support_comment = fields.String()
    documents = NestedWithKey(DocumentImageSchema, 'document_type', many_per_key=True)
    moderation = fields.Nested(AdminModerationSchema, dump_only=True)


class GetMerchantListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(AdminMerchantSchema(partial=True), many=True)


class SupportCommentMerchantRequestSchema(BaseSchema):
    support_comment = fields.String(required=True)


class BlockMerchantRequestSchema(BaseSchema):
    terminate_contract = fields.Boolean(default=True)


class NextPageSchema(BaseSchema):
    keyset = fields.Nested(GetMerchantListKeysetEntitySchema)


class GetMerchantListResponseV2DataSchema(BaseSchema):
    merchants = fields.Nested(AdminMerchantSchema, many=True)
    next_page = fields.Nested(NextPageSchema, dump_to='next', attribute='next')
    total = fields.Integer()
    found = fields.Integer()


class GetMerchantListResponseV2Schema(SuccessResponseSchema):
    data = fields.Nested(GetMerchantListResponseV2DataSchema)


get_merchant_list_request_schema = GetMerchantListRequestSchema()
get_merchant_list_request_schema_v2 = GetMerchantListRequestV2Schema()
get_merchant_list_response_schema = GetMerchantListResponseSchema()
get_merchant_list_response_schema_v2 = GetMerchantListResponseV2Schema()
block_merchant_request_schema = BlockMerchantRequestSchema()
support_comment_request_schema = SupportCommentMerchantRequestSchema()
