from dataclasses import asdict

from marshmallow import fields, pre_dump, validate
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import BasePaginatedRequestSchema, BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.fields import QueryParamList
from mail.payments.payments.api.schemas.merchant import ServiceSchema
from mail.payments.payments.api.schemas.transaction import TransactionSchema
from mail.payments.payments.core.entities.enums import PayStatus, TransactionStatus


class GetTransactionListRequestSchema(BasePaginatedRequestSchema):
    sort_by = fields.String(
        validate=validate.OneOf(['tx_id', 'updated', 'created', 'status', 'order_pay_status']),
        default='tx_id'
    )
    desc = fields.Boolean(default=True)

    uid = fields.Integer(load_from='merchant_uid')
    tx_id = fields.Integer()
    email = fields.String()
    order_id = fields.Integer()
    statuses = QueryParamList(EnumField(TransactionStatus, by_value=True))
    order_pay_statuses = QueryParamList(EnumField(PayStatus, by_value=True))
    lower_created_dt = fields.DateTime()
    upper_created_dt = fields.DateTime()
    lower_updated_dt = fields.DateTime()
    upper_updated_dt = fields.DateTime()
    customer_uid = fields.Integer()
    services = QueryParamList(fields.Integer())


class AdminServiceMerchantSchema(BaseSchema):
    service_merchant_id = fields.Integer()
    service = fields.Nested(ServiceSchema)


class AdminTransactionSchema(TransactionSchema):
    user_email = fields.String()
    customer_uid = fields.Integer()
    order_pay_status = EnumField(PayStatus, attribute='order.pay_status', by_value=True, dump_only=True)
    trust_receipt_download_url = fields.String()
    trust_receipt_view_url = fields.String()
    service_merchant = fields.Nested(AdminServiceMerchantSchema)

    @pre_dump
    def pre_dump(self, data):
        service_merchant = data.order.service_merchant
        data = asdict(data)
        return {
            'service_merchant': service_merchant,
            **data
        }


class GetTransactionListResponseSchema(SuccessResponseSchema):
    data = fields.Nested(AdminTransactionSchema, many=True)


class GetTransactionListResponseV2DataSchema(BaseSchema):
    transactions = fields.Nested(AdminTransactionSchema, many=True)
    total = fields.Integer()
    found = fields.Integer()


class GetTransactionListResponseV2Schema(SuccessResponseSchema):
    data = fields.Nested(GetTransactionListResponseV2DataSchema)


get_transaction_list_request_schema = GetTransactionListRequestSchema()
get_transaction_list_response_schema = GetTransactionListResponseSchema()
get_transaction_list_response_schema_v2 = GetTransactionListResponseV2Schema()
