from typing import Callable, Iterable, Optional

from aiohttp import web
from aiohttp.web_middlewares import normalize_path_middleware

from sendr_aiohttp import create_apispec, setup_swagger_route
from sendr_aiopg.engine.single import CustomEngine

from mail.payments.payments import __version__
from mail.payments.payments.api.app import DBEngineApplication
from mail.payments.payments.api.middlewares import (
    middleware_header_cloud, middleware_logging_adapter, middleware_response_formatter, middleware_stats
)
from mail.payments.payments.api.routes.base import UrlDispatcher
from mail.payments.payments.api_sdk.middlewares import middleware_request_checker
from mail.payments.payments.api_sdk.routes.monitoring import MONITORING_URLS
from mail.payments.payments.api_sdk.routes.v1 import V1_ROUTES
from mail.payments.payments.conf import settings
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.interactions.base import AbstractInteractionClient
from mail.payments.payments.storage.writers import PaymentsPushers


class SDKApplication(DBEngineApplication):
    _urls = (
        V1_ROUTES,
        MONITORING_URLS,
    )

    _middlewares: Iterable[Callable] = (
        normalize_path_middleware(remove_slash=True, append_slash=False),
        middleware_header_cloud,
        middleware_logging_adapter,
        middleware_stats,
        middleware_response_formatter,
        middleware_request_checker,
    )

    def __init__(self, db_engine: Optional[CustomEngine] = None, crypto: Optional[Crypto] = None):
        super().__init__(
            db_engine=db_engine,
            router=UrlDispatcher(),
            middlewares=self._middlewares,
        )

        self.crypto = crypto or Crypto.from_file(settings.CRYPTO_KEYS_FILE)
        self.pushers = PaymentsPushers()

        self.add_routes()
        self.add_sentry()

        self.on_cleanup.append(self.close_interaction_connector)
        self.on_cleanup.append(self.pushers.close)

    def add_routes(self) -> None:
        for routes in self._urls:
            self.router.add_routes(routes)
        if settings.SWAGGER_ENABLED:
            spec = create_apispec(
                app=self,
                title='Payments Managing Panel API',
                version=__version__,
            )
            setup_swagger_route(
                app=self,
                spec=spec,
                swagger_route='/api/swagger.json',
                swagger_route_name='swagger',
            )

    def add_sentry(self):
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            self.on_cleanup.append(sentry_init(settings.SENTRY_DSN, release=__version__))

    async def close_interaction_connector(self, _: web.Application) -> None:
        await AbstractInteractionClient.close_connector()
