from sendr_aiohttp.handler import request_schema, response_schema

from mail.payments.payments.api.schemas.base import success_response_schema
from mail.payments.payments.api.schemas.internal.order import post_internal_order_resized_request_schema
from mail.payments.payments.api.schemas.path import order_request_schema
from mail.payments.payments.api_sdk.handlers.base import BaseSDKHandler
from mail.payments.payments.api_sdk.schemas.order import (
    order_response_schema, post_order_request_schema, post_refund_request_schema, put_order_request_schema,
    refund_response_schema
)
from mail.payments.payments.api_sdk.schemas.path import clear_unhold_order_request_schema, refund_request_schema
from mail.payments.payments.core.actions.order.activate import CoreActivateOrderAction
from mail.payments.payments.core.actions.order.cancel import CoreCancelOrderAction
from mail.payments.payments.core.actions.order.clear_unhold import CoreScheduleClearUnholdOrderAction
from mail.payments.payments.core.actions.order.create_or_update import CoreCreateOrUpdateOrderAction
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.actions.order.pay_offline import CorePayOfflineOrderAction
from mail.payments.payments.core.actions.order.refund import CoreCreateRefundAction
from mail.payments.payments.core.entities.enums import OrderSource


class CreateOrderHandler(BaseSDKHandler):
    @request_schema(post_order_request_schema, location='json')
    @response_schema(order_response_schema)
    async def post(self):
        data = await self.run_action(
            CoreCreateOrUpdateOrderAction,
            {
                'source': OrderSource.SDK_API,
                **(await self.get_data())
            },
        )
        return self.make_response({'data': data}, order_response_schema)


class OrderHandler(BaseSDKHandler):
    @request_schema(order_request_schema, location='match_info')
    @response_schema(order_response_schema)
    async def get(self):
        data = await self.run_action(CoreGetOrderAction, await self.get_data())
        return self.make_response({'data': data}, order_response_schema)

    @request_schema(order_request_schema, location='match_info')
    @request_schema(put_order_request_schema, location='json')
    @response_schema(order_response_schema)
    async def put(self):
        data = await self.run_action(
            CoreCreateOrUpdateOrderAction,
            {
                'source': OrderSource.SDK_API,
                **(await self.get_data())
            }
        )
        return self.make_response({'data': data}, order_response_schema)


class ClearUnholdOrderHandler(BaseSDKHandler):
    @request_schema(post_internal_order_resized_request_schema, location='json')
    @request_schema(clear_unhold_order_request_schema, location='match_info')
    @response_schema(success_response_schema)
    async def post(self):
        await self.run_action(CoreScheduleClearUnholdOrderAction, await self.get_data())
        return self.make_response({'data': {}}, success_response_schema)


class DeactivateOrderHandler(BaseSDKHandler):
    @request_schema(order_request_schema, location='match_info')
    @response_schema(order_response_schema)
    async def post(self):
        data = await self.run_action(CoreActivateOrderAction, {**(await self.get_data()), 'active': False})
        return self.make_response({'data': data}, order_response_schema)


class PayOfflineOrderHandler(BaseSDKHandler):
    @request_schema(order_request_schema, location='match_info')
    @response_schema(order_response_schema)
    async def post(self):
        data = await self.run_action(CorePayOfflineOrderAction, await self.get_data())
        return self.make_response({'data': data}, order_response_schema)


class CancelOrderHandler(BaseSDKHandler):
    @request_schema(order_request_schema, location='match_info')
    @response_schema(order_response_schema)
    async def post(self):
        data = await self.run_action(CoreCancelOrderAction, await self.get_data())
        return self.make_response({'data': data}, order_response_schema)


class RefundHandler(BaseSDKHandler):
    @request_schema(refund_request_schema, location='match_info')
    @response_schema(refund_response_schema)
    async def get(self):
        data = await self.get_data()
        refund = await self.run_action(CoreGetOrderAction, {
            'uid': data['uid'],
            'original_order_id': data['order_id'],
            'order_id': data['refund_id']
        })
        return self.make_response({'data': refund}, refund_response_schema)

    @request_schema(post_refund_request_schema, location='json')
    @request_schema(order_request_schema, location='match_info')
    @response_schema(refund_response_schema)
    async def post(self):
        data = await self.run_action(CoreCreateRefundAction, await self.get_data())
        return self.make_response({'data': data}, refund_response_schema)
