from marshmallow import fields, validate
from marshmallow_enum import EnumField

from mail.payments.payments.api.schemas.base import CURRENCY_RUB, BaseSchema, SuccessResponseSchema
from mail.payments.payments.api.schemas.mixins import PayTokenMixin
from mail.payments.payments.api.schemas.order import OrderItemSchema, PostOrderRequestSchema
from mail.payments.payments.core.entities.enums import PayStatus, RefundStatus, ShopType


class SdkPostOrderRequestSchema(PostOrderRequestSchema):
    class Meta:
        strict = True
        fields = (
            'autoclear',
            'caption',
            'default_shop_type',
            'description',
            'items',
            'meta',
            'offline_abandon_deadline',
            'pay_method',
            'paymethod_id',
            'return_url',
        )

    meta = fields.String(validate=validate.Length(max=1024), missing='', default='')


class SdkPutOrderRequestSchema(PostOrderRequestSchema):
    class Meta:
        strict = True
        fields = (
            'autoclear',
            'caption',
            'description',
            'items',
            'meta',
            'offline_abandon_deadline',
            'pay_method',
            'paymethod_id',
            'return_url',
        )

    meta = fields.String(validate=validate.Length(max=1024), missing='', default='')


class SdkPostRefundRequestSchema(PostOrderRequestSchema):
    class Meta:
        strict = True
        fields = ('caption', 'description', 'items', 'meta')

    meta = fields.String(validate=validate.Length(max=1024), missing='', default='')


class BaseSdkOrderSchema(BaseSchema):
    active = fields.Boolean()
    caption = fields.String()
    description = fields.String()
    order_id = fields.Integer()
    items = fields.Nested(OrderItemSchema, many=True, dump_only=True)
    revision = fields.Integer()
    shop_type = EnumField(ShopType, attribute='shop.shop_type', dump_only=True, dump_to='mode', by_value=True)
    meta = fields.String(attribute='data.meta')
    price = fields.Decimal(places=2, dump_only=True)
    currency = fields.String(validate=validate.Equal(CURRENCY_RUB), dump_only=True)
    pay_method = fields.String()
    offline_abandon_deadline = fields.DateTime()
    created = fields.DateTime()
    updated = fields.DateTime()


class SdkOrderSchema(PayTokenMixin, BaseSdkOrderSchema):
    autoclear = fields.Boolean()
    payments_url = fields.String()
    pay_status = EnumField(PayStatus, by_value=True)
    user_email = fields.String(dump_only=True, allow_none=True)


class SdkRefundSchema(BaseSdkOrderSchema):
    class Meta:
        strict = True
        exclude = ('pay_method', 'items.image')

    order_id = fields.Integer(attribute='original_order_id')
    refund_id = fields.Integer(attribute='order_id')
    refund_status = EnumField(RefundStatus, by_value=True)


class SdkOrderResponseSchema(SuccessResponseSchema):
    data = fields.Nested(SdkOrderSchema)


class SdkRefundResponseSchema(SuccessResponseSchema):
    data = fields.Nested(SdkRefundSchema)


post_order_request_schema = SdkPostOrderRequestSchema()
put_order_request_schema = SdkPutOrderRequestSchema()
order_response_schema = SdkOrderResponseSchema()

post_refund_request_schema = SdkPostRefundRequestSchema()
refund_response_schema = SdkRefundResponseSchema()
