import array

import click
from aiopg.sa import SAConnection
from sqlalchemy import select
from sqlalchemy.sql import Selectable

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.shop.get_or_ensure_default import GetOrEnsureDefaultShopAction
from mail.payments.payments.core.entities.enums import ShopType
from mail.payments.payments.core.entities.shop import Shop
from mail.payments.payments.storage.db.tables import merchants as t_merchants
from mail.payments.payments.utils.cli import action_command

AIOPG_WAITER_TIMEOUT = 2 * 60
settings.DATABASE['TIMEOUT'] = AIOPG_WAITER_TIMEOUT


def merchants_query() -> Selectable:
    """
    Мерчантов у нас пока не много, поэтому выбирем их всех для миграции.
    """
    return (
        select([t_merchants.c.uid])
        .distinct()
        .order_by(t_merchants.c.uid)
    )


@click.command()
@action_command
async def cli() -> None:
    await CreateDefaultShopForAllMerchantAction().run()


class CreateDefaultShopForAllMerchantAction(BaseDBAction):
    """
    Одноразовая команда для создания дефолтных магазинов для всех существующих продавцов,
    и проставления shop_id для уже созданных заказов.

    После миграции можно удалить команду, соответствующий тест,
    а также поправить код создания заказа - считать, что дефолтный мгазин есть всегда
    и поправить определение колонки shop_id на NOT NULL.
    """

    async def handle(self) -> None:
        db_connection: SAConnection = self.storage.conn
        click.echo('Get or create default shops command start')
        click.echo('Select all merchant UIDs')
        merchant_uids = array.array('Q')
        async for row in db_connection.execute(merchants_query()):
            merchant_uids.append(row['uid'])
        click.echo(f'Got {len(merchant_uids)} merchants')
        click.echo('Start updating orders per merchant')

        for uid in merchant_uids:
            click.echo(f'Start transaction for merchant {uid}')
            async with db_connection.begin():
                # each merchant's orders are updated in separate transaction per merchant
                default_shop: Shop = await GetOrEnsureDefaultShopAction(uid=uid).run()
                default_test_shop: Shop = await GetOrEnsureDefaultShopAction(
                    uid=uid,
                    default_shop_type=ShopType.TEST,
                ).run()
                click.echo(f'Get or create default PROD shop {default_shop.shop_id}')
                click.echo(f'Get or create default TEST shop {default_test_shop.shop_id}')
            click.echo(f'Done transaction for merchant {uid}')
