from datetime import date, datetime

import click
from click import ClickException

from mail.payments.payments.core.actions.merchant.create import CreateMerchantAction
from mail.payments.payments.core.actions.moderation import ScheduleMerchantModerationAction
from mail.payments.payments.core.actions.worker.moderation import UpdateModerationAction
from mail.payments.payments.core.entities.enums import FunctionalityType, MerchantType, PersonType
from mail.payments.payments.core.entities.functionality import PaymentsFunctionalityData
from mail.payments.payments.core.entities.moderation import ModerationResult
from mail.payments.payments.core.exceptions import CoreDataError
from mail.payments.payments.utils.cli import action_command
from mail.payments.payments.utils.helpers import print_done


@click.command()
@click.option('--uid', type=str)
@action_command
async def cli(uid: str) -> None:
    if uid is None:
        raise ClickException('missing --uid')

    merchant_data = {
        'uid': uid,
        'name': 'No name',
        'functionality': PaymentsFunctionalityData(),
        'addresses': [
            {
                'type': 'legal',
                'zip': '119021',
                'country': 'RUS',
                'city': 'Moscow',
                'street': 'Lva Tolstogo',
                'home': '16',
            },
            {
                'type': 'post',
                'zip': '119021',
                'country': 'RUS',
                'city': 'Moscow',
                'street': 'Lva Tolstogo',
                'home': '16',
            }
        ],
        'bank': {
            'account': '30232810100000000004',
            'bik': '044525974',
            'correspondent_account': '30101810145250000974',
            'name': 'Tinkoff'
        },
        'organization': {
            'type': MerchantType.OOO,
            'inn': '5043041353',
            'kpp': '504301001',
            'ogrn': '1234567890123',
            'site_url': 'sell.yandex.ru',
            'name': 'Yandex',
            'full_name': 'Hoofs & Horns',
            'english_name': 'HH',
            'schedule_text': 'с 8 до 20'
        },
        'persons': [
            {
                'type': PersonType.CEO,
                'phone': '+79251234567',
                'surname': 'Калашников',
                'name': 'Вячеслав',
                'patronymic': 'Алексеевич',
                'birth_date': date(1994, 4, 1),
                'email': 'slava@ya.ru',
            },
            {
                'type': PersonType.CONTACT,
                'phone': '+79251234568',
                'surname': 'Самойлов',
                'name': 'Яромир',
                'patronymic': 'Евгеньевич',
                'birth_date': date(1994, 4, 1),
                'email': 'yaromir@ya.ru',
            },
            {
                'type': PersonType.SIGNER,
                'phone': '+79251234569',
                'surname': 'Борисенко',
                'name': 'Дмитрий',
                'patronymic': 'Артёмович',
                'birth_date': date(1994, 4, 1),
                'email': 'dmitry@ya.ru',
            }
        ],
    }

    with print_done():
        try:
            merchant = await CreateMerchantAction(**merchant_data, skip_init_submerchant=True).run()  # type: ignore
        except CoreDataError as e:
            raise ClickException(str(e))

        moderation = await ScheduleMerchantModerationAction(
            merchant=merchant,
            skip_moderation_task=True,
            functionality_type=FunctionalityType.PAYMENTS,
        ).run()

        await UpdateModerationAction(
            moderation_result=ModerationResult(
                moderation_id=moderation.moderation_id,
                client_id=merchant.client_id,
                approved=True,
                unixtime=int(datetime.now().timestamp()),
                reasons=[1, 2],
                submerchant_id=None,
                reason=None,
            )
        ).run()
