import codecs
import csv
import sys
from datetime import datetime, timedelta
from decimal import ROUND_HALF_UP, Decimal

import click
from click import INT, STRING, ClickException
from pytz import timezone

from mail.payments.payments.core.actions.merchant.stats import MerchantsOrdersStatsAction
from mail.payments.payments.core.entities.enums import AcquirerType
from mail.payments.payments.utils.cli import action_command


@click.command()
@click.option('--date_from', type=STRING)
@click.option('--date_to', type=STRING)
@click.option('--acquirer', type=STRING)
@click.option('--service_id', type=INT, default=None)
@click.option(
    '--decimal_delimiter_comma',
    type=bool,
    default=True,
    help='Use "," as decimal delimiter instead of "."',
)
@action_command
async def cli(date_from=None, date_to=None, acquirer=None, service_id=None, decimal_delimiter_comma=True):
    """
    Возвращает статистику по мерчантам: сумма платежей, комиссия, возвраты
    --date_from и --date_to — фильтры по датам

    ВНИМАНИЕ! В формировании отчета используются заказы закрытые в период от date_from до date_to ВКЛЮЧИТЕЛЬНО.
    """
    tz = timezone('Europe/Moscow')

    closed_from = closed_to = None
    if date_from is not None:
        closed_from = tz.localize(datetime.strptime(date_from, '%Y-%m-%d'))
    if date_to is not None:
        closed_to = tz.localize(datetime.strptime(date_to, '%Y-%m-%d')) + timedelta(days=1)
    if acquirer is not None:
        try:
            acquirer = AcquirerType(acquirer)
        except ValueError:
            raise ClickException(f'Unknown value: {acquirer}. Allowed: {", ".join([x.value for x in AcquirerType])}')

    click.echo(
        f'\nGenerating merchant order stats using orders closed from {closed_from} and before {closed_to}...\n',
        file=sys.stderr,
    )
    rows = await MerchantsOrdersStatsAction(closed_from=closed_from,
                                            closed_to=closed_to,
                                            service_id=service_id,
                                            acquirer=acquirer).run()

    writer = csv.writer(sys.stdout, delimiter=';', quotechar='"', quoting=csv.QUOTE_ALL)
    current_kind = None

    for row in rows:
        if current_kind != row.orders_kind:
            if current_kind is not None:
                sys.stdout.write('\n')
            else:
                sys.stdout.write(codecs.BOM_UTF8.decode('utf-8'))
                writer.writerow(["parent_uid", "title", "sum", "commission"])
            current_kind = row.orders_kind

        order_sum = "%.2f" % row.orders_sum.quantize(Decimal('1.00'), rounding=ROUND_HALF_UP)
        commission = "%.2f" % row.commission.quantize(Decimal('1.00'), rounding=ROUND_HALF_UP)
        if decimal_delimiter_comma:
            order_sum = order_sum.replace('.', ',')
            commission = commission.replace('.', ',')
        writer.writerow([row.parent_uid or '-', row.name, order_sum, commission])
