import importlib

import click
import sqlalchemy as sa
from click import ClickException
from sqlalchemy.schema import CreateSequence, CreateTable
from sqlalchemy.sql.ddl import DDLElement

from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.utils.cli import action_command


class CreateEnum(DDLElement):
    __visit_name__ = "create_enum_type"

    def __init__(self, element: sa.Enum):
        self.element = element


@click.argument('entities')
@click.command()
@action_command
async def cli(entities: str) -> None:
    """
    Генерирует SQL DDL по моделькам из payments/storage/db/tables.py
    PYTHONPATH=. ./payments/manage.py sql_create orders,merchants ->
    CREATE TABLE payments.orders (
        uid BIGINT NOT NULL,
        order_id BIGINT NOT NULL,
        ...
    )
    """
    compilers = {
        sa.Table: CreateTable,
        sa.Sequence: CreateSequence,
        sa.Enum: CreateEnum
    }

    tables = importlib.import_module("mail.payments.payments.storage.db.tables")

    for entity in entities.split(','):
        entity = str(entity).strip()

        try:
            entity_instance = getattr(tables, entity)
            entity_type = type(entity_instance)
        except AttributeError:
            raise ClickException(f'{entity} not found')

        if entity_type in compilers:
            compiler = compilers[entity_type]
        else:
            raise ClickException(f'Unknown entity: {entity_type}')

        db_engine = BaseAction.context.db_engine
        entity_compiler = compiler(entity_instance).compile(db_engine)
        print(str(entity_compiler).strip() + ';')
