from dataclasses import fields
from typing import Any, Optional

import click
from click import ClickException

from mail.payments.payments.core.actions.merchant.update import UpdateMerchantAction
from mail.payments.payments.core.entities.enums import PersonType
from mail.payments.payments.core.entities.merchant import AddressData, BankData, OrganizationData, PersonData
from mail.payments.payments.core.exceptions import CoreDataError, CoreFieldError
from mail.payments.payments.utils.cli import action_command
from mail.payments.payments.utils.helpers import print_done


def _parse_arg_value(arg_value: str) -> dict:
    try:
        result = {param.split(':')[0]: param.split(':')[1] for param in arg_value.split(',')}
    except Exception:
        raise ClickException(f'Invalid argument: {arg_value}')
    if 'type' in result:
        raise ClickException('"type" cannot be changed')
    return result


def _get_fields_list_for_help(dataclass: Any) -> list:
    return [field.name for field in fields(dataclass) if field.name != 'type']


def _get_help(dataclass: Any) -> str:
    fields_str = ', '.join(_get_fields_list_for_help(dataclass))
    return f'key1:value1[,key2:value2]... Acceptable keys: {fields_str}'


@click.command()
@click.option('--uid', type=str, required=True)
@click.option('--name', type=str)
@click.option('--legal_address', type=str, help=_get_help(AddressData))
@click.option('--post_address', type=str, help=_get_help(AddressData))
@click.option('--bank', type=str, help=_get_help(BankData))
@click.option('--organization', type=str, help=_get_help(OrganizationData))
@click.option('--ceo_person', type=str, help=_get_help(PersonData))
@click.option('--contact_person', type=str, help=_get_help(PersonData))
@click.option('--signer_person', type=str, help=_get_help(PersonData))
@action_command
async def cli(uid: str,
              name: Optional[str],
              bank: Optional[str],
              organization: Optional[str],
              legal_address: Optional[str],
              post_address: Optional[str],
              ceo_person: Optional[str],
              contact_person: Optional[str],
              signer_person: Optional[str]) -> None:
    merchant_data = {'uid': uid}

    if name is not None:
        merchant_data['name'] = name

    flat_data_params = [
        (bank, 'bank'),
        (organization, 'organization'),
    ]
    for arg, path in flat_data_params:
        if arg is not None:
            new_fields = _parse_arg_value(arg)
            merchant_data[path] = new_fields  # type: ignore

    data_with_key_params = [
        (legal_address, 'addresses', 'legal'),
        (post_address, 'addresses', 'post'),
        (ceo_person, 'persons', PersonType.CEO),
        (contact_person, 'persons', PersonType.CONTACT),
        (signer_person, 'persons', PersonType.SIGNER),
    ]
    for arg, path, type in data_with_key_params:
        if arg is not None:
            new_fields = _parse_arg_value(arg)
            merchant_data.setdefault(path, {})[type] = new_fields  # type: ignore

    with print_done():
        try:
            await UpdateMerchantAction(params=merchant_data,
                                       check_not_registered=False,
                                       send_notifications=True).run()
        except CoreFieldError as e:
            raise ClickException(str(e.fields))
        except CoreDataError as e:
            assert e.MESSAGE
            raise ClickException(e.MESSAGE)
