from typing import Optional

from mail.payments.payments.core.actions.arbitrage.get_order import GetOrderWithCurrentArbitrage
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.arbitrage import Arbitrage
from mail.payments.payments.core.entities.enums import OrderKind
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import (
    ActiveArbitrageAlreadyExistError, MerchantNotConnectedToDialogsError, OrderAnonymousError, OrderInvalidKind,
    OrderMustBePaidError
)


class ConsultationArbitrageAction(BaseMerchantAction):
    skip_data = True
    transact = True

    def __init__(self, uid: int, order_id: int, customer_uid: Optional[int], merchant: Optional[Merchant] = None):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id = order_id
        self.customer_uid = customer_uid

    async def handle(self) -> Arbitrage:
        assert self.uid and self.merchant

        order: Order = await GetOrderWithCurrentArbitrage(
            uid=self.uid,
            order_id=self.order_id,
            customer_uid=self.customer_uid,
            for_update=True
        ).run()

        assert order.order_id

        if order.kind != OrderKind.PAY or order.customer_subscription_id is not None:
            raise OrderInvalidKind
        elif order.current_arbitrage is not None:
            raise ActiveArbitrageAlreadyExistError
        elif not order.is_already_paid:
            raise OrderMustBePaidError
        elif not self.merchant.dialogs_org_id:
            raise MerchantNotConnectedToDialogsError
        elif not order.customer_uid:
            raise OrderAnonymousError

        description = f'Арбитраж по заказу №{order.order_id}'
        consultation = await self.clients.floyd.create_consultation(self.merchant.dialogs_org_id, description, order)

        consultation_id = consultation['id']
        chat_id = consultation['chats']['clients']['chat_id']

        return await self.storage.arbitrage.create(
            Arbitrage(
                uid=order.uid,
                order_id=order.order_id,
                chat_id=chat_id,
                consultation_id=consultation_id
            )
        )
