from typing import Optional

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.arbitrage.get_order import GetOrderWithCurrentArbitrage
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.entities.arbitrage import Arbitrage
from mail.payments.payments.core.entities.enums import ArbitrageStatus
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import ActiveArbitrageAbsentError


class StartEscalateAction(BaseMerchantAction):
    transact = True
    skip_moderation = True
    action_name = 'start_escalate'

    def __init__(self, uid: int, order_id: int):
        super().__init__(uid=uid)
        self.order_id = order_id

    async def handle(self) -> None:
        assert self.uid and self.order_id
        order: Order = await CoreGetOrderAction(uid=self.uid, order_id=self.order_id).run()
        arbitrage = await self.storage.arbitrage.get_current(self.uid, self.order_id, for_update=True)
        self.logger.context_push(uid=self.uid, order_id=self.order_id, customer_uid=order.customer_uid)

        if not arbitrage or arbitrage.status != ArbitrageStatus.ESCALATE:
            self.logger.info('Arbitrage is absent.')
            return

        self.logger.context_push(arbitrage_id=arbitrage.arbitrage_id)
        assert self.merchant and self.merchant.dialogs_org_id and arbitrage.chat_id
        org_id = self.merchant.dialogs_org_id
        assert org_id is not None
        assert arbitrage.consultation_id is not None
        await self.clients.floyd.update_chatbar(
            org_id=org_id,
            consultation_id=arbitrage.consultation_id,
            role_id='client',
            chatbar={'title': None, 'button': None},
        )

        # Скачиваем сообщения кастомера с мерчантом
        messages = await self.clients.floyd.get_consultation_history(org_id, arbitrage.consultation_id)

        # Создаем чат мерчант-арбитр
        arbitrage.arbiter_chat_id, _ = await self.clients.floyd.create_organizations_chat(
            org_id,
            settings.FLOYD_ARBITRAGE_ORG_ID
        )

        # Создаем арбитраж
        arbitrage.escalate_id = await self.clients.arbiter.create_arbitrage(self.merchant, order, arbitrage, messages)
        arbitrage = await self.storage.arbitrage.save(arbitrage)

        assert arbitrage.consultation_id is not None
        # Вызываем кастомный экшн в сценарии, чтобы перерисовать интерфейс
        await self.clients.floyd.custom_action(
            org_id,
            arbitrage.consultation_id,
            'startArbitrage',
            {'arbitrage_org_id': settings.FLOYD_ARBITRAGE_ORG_ID}
        )


class EscalateArbitrageAction(BaseMerchantAction):
    skip_data = True
    transact = True

    def __init__(self, uid: int, order_id: int, customer_uid: Optional[int], merchant: Optional[Merchant] = None):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id = order_id
        self.customer_uid = customer_uid

    async def handle(self) -> Arbitrage:
        assert self.uid and self.merchant
        self.logger.context_push(
            uid=self.uid, order_id=self.order_id, customer_uid=self.customer_uid
        )

        order: Order = await GetOrderWithCurrentArbitrage(
            uid=self.uid,
            order_id=self.order_id,
            customer_uid=self.customer_uid,
            for_update=True
        ).run()

        assert order.order_id

        if order.current_arbitrage is None or order.current_arbitrage.status != ArbitrageStatus.CONSULTATION:
            self.logger.info('Consultation is absent.')
            raise ActiveArbitrageAbsentError

        await StartEscalateAction(uid=self.uid, order_id=self.order_id).run_async()
        self.logger.info('Escalating was scheduled.')

        order.current_arbitrage.status = ArbitrageStatus.ESCALATE
        order.current_arbitrage = await self.storage.arbitrage.save(order.current_arbitrage)

        return order.current_arbitrage
