from typing import Optional

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.actions.order.get import CoreGetOrderAction
from mail.payments.payments.core.entities.enums import OrderKind
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import OrderInvalidKind


class GetOrderWithCurrentArbitrage(BaseMerchantAction):
    skip_data = True

    def __init__(self,
                 order_id: int,
                 customer_uid: Optional[int] = None,
                 for_update: bool = False,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None):
        super().__init__(uid=uid, merchant=merchant)
        self.order_id = order_id
        self.customer_uid = customer_uid
        self._for_update = for_update

    async def handle(self) -> Order:
        assert self.uid

        order = await CoreGetOrderAction(
            uid=self.uid,
            order_id=self.order_id,
            customer_uid=self.customer_uid,
            for_update=self._for_update,
            select_customer_subscription=None
        ).run()

        if order.kind != OrderKind.PAY or order.customer_subscription_id is not None:
            raise OrderInvalidKind

        order.merchant = self.merchant
        order.current_arbitrage = await self.storage.arbitrage.get_current(
            order.uid,
            order.order_id,
            for_update=self._for_update
        )

        return order
