from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.entities.arbitrage import Arbitrage
from mail.payments.payments.core.entities.enums import ArbitrageVerdict, RefundStatus
from mail.payments.payments.storage.exceptions import OrderNotFound


class NotifyArbitrageAction(BaseDBAction):
    action_name = 'notify_arbitrage_action'

    def __init__(self, arbitrage_id: int):
        super().__init__()
        self.arbitrage_id = arbitrage_id

    async def handle(self) -> None:
        self.logger.context_push(arbitrage_id=self.arbitrage_id)

        try:
            arbitrage = await self.storage.arbitrage.get(self.arbitrage_id)
        except Arbitrage.DoesNotExist:
            self.logger.warning('Arbitrage does not exist.')
            return

        if arbitrage.escalate_id is None:
            self.logger.warning('Arbitrage was not created in ABO.')
            return
        elif arbitrage.verdict is None:
            self.logger.warning('Arbitrage without verdict.')
            return

        merchant = await GetMerchantAction(uid=arbitrage.uid, skip_data=True).run()

        await self.clients.floyd.custom_action(
            merchant.dialogs_org_id,
            arbitrage.consultation_id,
            'finishArbitrage',
            {
                'org_id': merchant.dialogs_org_id,
                'verdict': arbitrage.verdict.value
            }
        )

        if arbitrage.verdict == ArbitrageVerdict.REFUND:
            try:
                refund = await self.storage.order.get(arbitrage.uid, arbitrage.refund_id)
            except (OrderNotFound, RuntimeError):  # Перед run_async есть проверка, защита от ручного вмешательства
                self.logger.warning('Refund not found.')
                return

            if refund.refund_status != RefundStatus.COMPLETED:
                self.logger.warning('Refund is not completed.')
                return

            await self.clients.arbiter.execute_verdict(conversation_id=arbitrage.escalate_id)
