from __future__ import annotations

from asyncio import CancelledError
from itertools import chain
from typing import Awaitable, ClassVar, FrozenSet, Optional, Tuple, TypeVar, Union

from aiohttp.client_exceptions import ClientConnectionError

import sendr_core.action
from sendr_interactions import exceptions as interaction_errors

from mail.payments.payments.core.context import CoreContext
from mail.payments.payments.core.entities.enums import MerchantRole
from mail.payments.payments.core.entities.merchant_user import MerchantUser
from mail.payments.payments.core.exceptions import (
    BaseCoreError, CoreFailError, CoreInteractionAlreadyExist, CoreInteractionConnectionTimeoutError,
    CoreInteractionFatalError, CoreInteractionNotFound, CoreInteractionRequestError, CoreInteractionResponseError,
    CoreInteractionTimeoutError, CoreMerchantUserNotAuthorizedError, CoreRevisionMismatch
)
from mail.payments.payments.file_storage import FileStorage
from mail.payments.payments.http_helpers.crypto import Crypto
from mail.payments.payments.interactions import InteractionClients
from mail.payments.payments.storage.writers import PaymentsPushers

_T = TypeVar('_T')


class BaseAction(sendr_core.action.BaseAction):
    context = CoreContext()
    required_merchant_roles: ClassVar[Optional[Tuple[MerchantRole, ...]]] = None

    def __init__(self):
        super().__init__()
        self._clients: InteractionClients = InteractionClients(
            logger=self.logger,
            request_id=self.request_id,
            pushers=self.context.pushers,
        )
        self._file_storage = FileStorage()

    @property
    def clients(self) -> InteractionClients:
        return self._clients

    @property
    def pushers(self) -> PaymentsPushers:
        return self.context.pushers

    @property
    def crypto(self) -> Crypto:
        return self.context.crypto

    @property
    def file_storage(self) -> FileStorage:
        return self._file_storage

    @property
    def merchant_user(self) -> Optional[MerchantUser]:
        return self.context.merchant_user

    @property
    def allowed_roles(self) -> Union[FrozenSet[Optional[MerchantRole]], FrozenSet[MerchantRole]]:
        if self.required_merchant_roles is None:
            return frozenset(chain((None,), MerchantRole))
        return frozenset(self.required_merchant_roles).union(
            *(role.superroles for role in self.required_merchant_roles)
        )

    def is_allowed_role(self, role: Optional[MerchantRole]) -> bool:
        return role in self.allowed_roles

    async def load_merchant_user(self) -> None:
        raise NotImplementedError

    async def check_merchant_user_roles(self) -> None:
        if self.required_merchant_roles is None:
            return
        await self.load_merchant_user()
        if self.context.merchant_user is None:
            return
        if not self.is_allowed_role(self.context.merchant_user.user_merchant_role):
            raise CoreMerchantUserNotAuthorizedError(self.allowed_roles)

    @staticmethod
    def _raise_response_error(exc: interaction_errors.InteractionResponseError) -> None:
        if exc.status_code >= 500:
            raise CoreInteractionFatalError
        if exc.status_code == 404:
            raise CoreInteractionNotFound
        if exc.response_status == 'revision_mismatch':
            raise CoreRevisionMismatch
        if exc.status_code == 409:
            raise CoreInteractionAlreadyExist
        if exc.status_code >= 400:
            raise CoreInteractionRequestError
        raise CoreInteractionResponseError

    async def _secure_run(self, coro: Awaitable[_T]) -> _T:  # type: ignore
        try:
            return await coro
        except BaseCoreError:
            raise
        except ClientConnectionError:
            raise CoreInteractionConnectionTimeoutError
        except interaction_errors.InteractionTimeoutError:
            raise CoreInteractionTimeoutError
        except interaction_errors.InteractionBadResponse:
            raise CoreInteractionResponseError
        except interaction_errors.InteractionResponseError as exc:
            self._raise_response_error(exc)
        except interaction_errors.BaseInteractionError:
            raise CoreInteractionFatalError
        except CancelledError:
            raise
        except Exception:
            self.logger.exception('Core failed')
            raise CoreFailError

    async def _run(self):
        with self.logger:
            self.logger.context_push(action=self.__class__.__name__)
            async with self.clients:
                await self._secure_run(self.check_merchant_user_roles())
                return await super()._run()
