from copy import deepcopy
from datetime import date
from typing import ClassVar, Iterable, List, Optional, cast

from sendr_utils import alist

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.mixins.moderation import MerchantModerationMixin
from mail.payments.payments.core.entities.enums import (
    FunctionalityType, MerchantDraftPolicy, MerchantStatus, MerchantType
)
from mail.payments.payments.core.entities.functionality import (
    Functionalities, PaymentsFunctionalityData, YandexPayFunctionalityData
)
from mail.payments.payments.core.entities.merchant import (
    AddressData, BankData, Merchant, MerchantData, OrganizationData, PersonData, PersonType
)
from mail.payments.payments.core.entities.merchant_oauth import MerchantOAuth
from mail.payments.payments.core.exceptions import (
    CoreActionDenyError, MerchantIsAlreadyRegistered, MerchantNotFoundError
)
from mail.payments.payments.interactions.balance.entities import Person
from mail.payments.payments.storage.exceptions import MerchantNotFound, MerchantPreregistrationNotFound


class BaseMerchantAction(MerchantModerationMixin, BaseDBAction):
    for_update: ClassVar[bool] = False
    allow_none: ClassVar[bool] = False
    manual_load: ClassVar[bool] = False

    # it's an error to skip parent and not skip data
    skip_parent: ClassVar[bool] = False
    skip_data: ClassVar[bool] = False
    skip_moderation: ClassVar[bool] = False
    skip_oauth: ClassVar[bool] = False
    skip_preregistration: ClassVar[bool] = True
    skip_functionalities: ClassVar[bool] = False

    check_moderation_approved: ClassVar[bool] = False
    check_moderation_disapproved: ClassVar[bool] = False

    draft_policy: ClassVar[MerchantDraftPolicy] = MerchantDraftPolicy.MERCHANT_DRAFT_FORBIDDEN

    uid: Optional[int] = None
    merchant: Optional[Merchant] = None

    def __init__(self,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__()
        self.uid = uid
        self.merchant = merchant

    async def _load_parent(self) -> None:
        assert self.merchant
        if self.merchant.parent_uid is None:
            self.merchant.load_parent()
            return
        try:
            self.merchant.parent = await self.storage.merchant.get(self.merchant.parent_uid)
        except MerchantNotFound:
            with self.logger:
                self.logger.context_push(
                    uid=self.merchant.uid,
                    parent_uid=self.merchant.parent_uid,
                )
                self.logger.error('Merchant parent does not exist')
            raise MerchantNotFoundError(uid=self.merchant.uid)
        self.merchant.load_parent()

    async def _load_data(self) -> None:
        assert self.merchant
        self.merchant.load_data()

    async def _load_moderation(self) -> None:
        assert self.merchant
        self.merchant.moderations = {}
        for functionality_type in FunctionalityType:
            self.merchant.moderations[functionality_type] = await self.get_moderation_data(
                self.merchant, functionality_type=functionality_type,
            )
        self.merchant.moderation = self.merchant.moderations.get(FunctionalityType.PAYMENTS)

    async def _load_oauth(self) -> None:
        assert self.merchant
        oauth: List[MerchantOAuth]

        oauth = await alist(self.storage.merchant_oauth.find_by_uid(self.merchant.uid))

        self.merchant.oauth = oauth

    async def _load_functionalities(self) -> None:
        assert self.merchant
        functionalities = await self.storage.functionality.find_by_uid(self.merchant.uid)
        functionalities_dict = {
            functionality.functionality_type: functionality.data
            for functionality in functionalities
        }

        self.merchant.functionalities = Functionalities(
            payments=cast(
                Optional[PaymentsFunctionalityData],
                functionalities_dict.get(FunctionalityType.PAYMENTS),
            ),
            yandex_pay=cast(
                Optional[YandexPayFunctionalityData],
                functionalities_dict.get(FunctionalityType.YANDEX_PAY),
            ),
        )

    async def _load_registration(self) -> None:
        assert self.merchant
        try:
            self.merchant.preregistration = await self.storage.merchant_preregistration.get(uid=self.merchant.uid)
        except MerchantPreregistrationNotFound:
            pass

    def _is_merchant_draft(self) -> bool:
        assert self.merchant is not None
        if self.merchant.status == MerchantStatus.DRAFT:
            if self.draft_policy == MerchantDraftPolicy.MERCHANT_DRAFT_FORBIDDEN:
                raise CoreActionDenyError
            return True
        else:
            if self.draft_policy == MerchantDraftPolicy.MERCHANT_DRAFT_REQUIRED:
                raise MerchantIsAlreadyRegistered
            return False

    async def _fetch_merchant_from_db(self,
                                      uid: Optional[int] = None,
                                      merchant: Optional[Merchant] = None,
                                      token: Optional[str] = None,
                                      for_update: Optional[bool] = None,
                                      ) -> None:
        for_update = self.for_update if for_update is None else for_update
        try:
            if merchant is not None:
                self.merchant = merchant
            elif uid is not None:
                self.merchant = await self.storage.merchant.get(uid, for_update=for_update)
            elif token is not None:
                self.merchant = await self.storage.merchant.find_by_token(token, for_update=for_update)
            else:
                raise RuntimeError('No merchant or merchant_uid is provided')
        except MerchantNotFound:
            pass

    async def _load_merchant(self,
                             uid: Optional[int] = None,
                             merchant: Optional[Merchant] = None,
                             token: Optional[str] = None,
                             for_update: Optional[bool] = None,
                             skip_parent: bool = False,
                             skip_data: bool = False,
                             skip_moderation: bool = False,
                             skip_oauth: bool = False,
                             skip_preregistration: bool = True,
                             skip_functionalities: bool = False,
                             ) -> None:
        await self._fetch_merchant_from_db(uid=uid, merchant=merchant, token=token, for_update=for_update)

        if self.merchant is None:
            if self.allow_none:
                return
            raise MerchantNotFoundError(uid=uid)
        elif self._is_merchant_draft():
            return

        if not skip_parent and not self.merchant.parent_loaded:
            await self._load_parent()
        if not skip_data and not self.merchant.data_loaded:
            # TODO: remove skip_data parameter and always invoke self.merchant.load_data()
            await self._load_data()
        if not skip_moderation:
            await self._load_moderation()
        if not skip_oauth:
            await self._load_oauth()
        if not skip_preregistration:
            await self._load_registration()
        if not skip_functionalities:
            await self._load_functionalities()

    async def load_merchant(self,
                            uid: Optional[int] = None,
                            merchant: Optional[Merchant] = None,
                            token: Optional[str] = None,
                            skip_parent: Optional[bool] = None,
                            skip_data: Optional[bool] = None,
                            skip_moderation: Optional[bool] = None,
                            skip_oauth: Optional[bool] = None,
                            skip_preregistration: Optional[bool] = None,
                            skip_functionalities: Optional[bool] = None,
                            for_update: Optional[bool] = None,
                            ) -> None:
        skip_parent = self.skip_parent if skip_parent is None else skip_parent
        skip_data = self.skip_data if skip_data is None else skip_data
        skip_moderation = self.skip_moderation if skip_moderation is None else skip_moderation
        skip_oauth = self.skip_oauth if skip_oauth is None else skip_oauth
        skip_preregistration = self.skip_preregistration if skip_preregistration is None else skip_preregistration
        skip_functionalities = self.skip_functionalities if skip_functionalities is None else skip_functionalities
        for_update = self.for_update if for_update is None else for_update

        if skip_parent and not skip_data:
            raise RuntimeError('Cannot load data without parent loaded')
        if merchant is None and token is None and uid is None:
            raise RuntimeError('No merchant or merchant_uid or token is provided')

        await self._load_merchant(
            uid=uid,
            merchant=merchant,
            token=token,
            skip_parent=skip_parent,
            skip_data=skip_data,
            skip_moderation=skip_moderation,
            skip_oauth=skip_oauth,
            skip_preregistration=skip_preregistration,
            skip_functionalities=skip_functionalities,
            for_update=for_update,
        )

        if self.check_moderation_approved:
            await self.require_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)
        if self.check_moderation_disapproved:
            await self.require_no_moderation(self.merchant, functionality_type=FunctionalityType.PAYMENTS)

        self.logger.context_push(uid=self.merchant.uid if self.merchant is not None else None)

    async def pre_handle(self) -> None:
        if not self.manual_load:
            await self.load_merchant(uid=self.uid, merchant=self.merchant)
        await super().pre_handle()

    @staticmethod
    def fill_merchant_from_person(merchant: Merchant, person: Person) -> bool:
        """
        :return: True if merchant.data has changed else False
        """
        data: Optional[MerchantData] = merchant.data
        assert data is not None

        old_data = deepcopy(data)

        data.addresses = [
            AddressData(
                type='legal',
                city=person.legal_address_city,
                country='RUS',
                home=person.legal_address_home,
                street=person.legal_address_street,
                zip=person.legal_address_postcode,
            )
        ]

        if person.has_post_address:
            data.addresses.append(AddressData(
                type='post',
                city=person.address_city,  # type: ignore
                country='RUS',
                home=person.address_home,
                street=person.address_street,  # type: ignore
                zip=person.address_postcode,  # type: ignore
            ))

        for item in [old_data, data]:
            if item.addresses:
                item.addresses = sorted(item.addresses, key=lambda address: address.type)

        correspondent_account: Optional[str] = None
        name: Optional[str] = None
        if data.bank is not None:
            correspondent_account = data.bank.correspondent_account
            name = data.bank.name
        data.bank = BankData(
            account=person.account,
            bik=person.bik,
            correspondent_account=correspondent_account,
            name=name,
        )

        birth_date: Optional[date] = None
        ceo: Optional[PersonData] = data.get_person(PersonType.CEO)
        if ceo is not None:
            birth_date = ceo.birth_date
        persons: Iterable[PersonData] = data.persons if data.persons is not None else []
        data.persons = [
            PersonData(
                type=PersonType.CEO,
                name=person.fname,
                email=person.email,
                phone=person.phone,
                surname=person.lname,
                patronymic=person.mname,
                birth_date=birth_date,
            ),
            *[person for person in persons if person.type != PersonType.CEO]
        ]

        for item in [old_data, data]:
            if item.persons:
                item.persons = sorted(item.persons, key=lambda person: person.type.value)

        org_type: Optional[MerchantType] = None
        english_name: Optional[str] = None
        site_url: Optional[str] = None
        schedule_text: Optional[str] = None
        description: Optional[str] = None
        if data.organization is not None:
            org_type = data.organization.type
            english_name = data.organization.english_name
            site_url = data.organization.site_url
            schedule_text = data.organization.schedule_text
            description = data.organization.description
        data.organization = OrganizationData(
            type=org_type,
            name=person.name,
            english_name=english_name,
            full_name=person.longname,
            inn=person.inn,
            kpp=person.kpp,
            ogrn=person.ogrn,
            site_url=site_url,
            schedule_text=schedule_text,
            description=description,
        )

        return data != old_data
