import re
from typing import Optional, Tuple

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import CoreFailError, CoreNotFoundError
from mail.payments.payments.interactions.balance.exceptions import BalanceContractNotFound, BalanceContractRuleViolation


class InitContractAction(BaseMerchantAction):
    transact = True
    skip_parent = False
    skip_data = True
    for_update = True

    async def handle(self) -> Tuple[Merchant, Optional[str]]:
        """
        Initialize Contract for Merchant
        Returns:
            Merchant, Contract External ID (if created contract, else None)
        """
        assert self.merchant is not None
        merchant: Merchant = self.merchant

        if merchant.contract_id or merchant.parent_uid:
            self.logger.context_push(
                contract_id=merchant.contract_id,
                parent_uid=merchant.parent_uid,
            )
            self.logger.info('Contract is not created: merchant has contract_id or parent_uid')
            return merchant, None

        assert (
            merchant.client_id is not None
            and merchant.person_id is not None
            and merchant.acquirer is not None
        )

        data_override = merchant.options.offer_settings.data_override

        try:
            contract_id, external_id = await self.clients.balance.create_offer(
                uid=merchant.uid,
                acquirer=merchant.acquirer,
                client_id=merchant.client_id,
                person_id=merchant.person_id,
                extra_params=data_override
            )
        except BalanceContractRuleViolation as exc:
            # If reusing of existing contract id from exception is enabled, drop 'person_id' filter from request.
            # Otherwise Balance will return empty list if 'person_id' has changed after merchant re-adding
            contracts = await self.clients.balance.get_client_contracts(
                client_id=merchant.client_id,
                person_id=None if settings.BALANCE_REUSE_CONTRACT_ID_FROM_ERROR else merchant.person_id,
                is_active=True
            )
            if settings.BALANCE_REUSE_CONTRACT_ID_FROM_ERROR:
                # Find contract with corresponding id or fallback to default behaviour if couldn't parse id
                # from exception
                existing_id = self._try_get_contract_id_from_exception(exc)
                contracts = [c for c in contracts if str(c['ID']) == existing_id] if existing_id else []
            if not contracts:
                self.logger.context_push(client_id=merchant.client_id, person_id=merchant.person_id)
                what_happened = 'Failed to create Contract and Balance returns no active contracts for Person'
                self.logger.exception(what_happened)
                raise CoreFailError(what_happened)  # insane situation
            contract = contracts[0]
            contract_id, external_id = contract['ID'], contract['EXTERNAL_ID']

        merchant.contract_id = f'{contract_id}'
        merchant = await self.storage.merchant.save(self.merchant)

        merchant.load_parent()  # no parent
        self.logger.context_push(contract_id=merchant.contract_id, revision=merchant.revision)
        self.logger.info('Merchant updated: contract_id')

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=OperationKind.INIT_CONTRACT,
            arguments={'contract_id': merchant.contract_id},
        ))

        self.merchant = merchant
        return self.merchant, external_id

    def _try_get_contract_id_from_exception(self, exc: BalanceContractRuleViolation) -> Optional[str]:
        pattern = r"Rule violation: 'Для клиента с id=\S+ уже существуют договоры с такими типом, " \
                  r"фирмой или сервисом: (\S+)'"
        m = re.match(pattern, exc.message)
        return m.group(1) if m else None


class TerminateContractAction(BaseMerchantAction):
    transact = True
    skip_data = True
    skip_parent = False
    for_update = True

    async def handle(self) -> Merchant:
        self.merchant: Merchant
        if not self.merchant.contract_id or self.merchant.parent_uid:
            self.logger.context_push(
                contract_id=self.merchant.contract_id,
                parent_uid=self.merchant.parent_uid,
            )
            self.logger.info('Contract is not terminated: merchant has no contract_id or has parent_id')
            return self.merchant

        try:
            await self.clients.balance.create_collateral(self.merchant.uid, self.merchant.contract_id)
        except BalanceContractNotFound as e:
            raise CoreNotFoundError(message=e.MESSAGE)
        self.merchant.contract_id = None
        self.merchant = await self.storage.merchant.save(self.merchant)
        self.merchant.load_parent()  # no parent
        self.logger.context_push(revision=self.merchant.revision)
        self.logger.info('Merchant updated: contract_id')

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=OperationKind.TERMINATE_CONTRACT,
            arguments={'contract_id': None},
        ))

        return self.merchant
