from typing import Optional, Tuple

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.get import GetOrderByCustomerSubscriptionIdAction
from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.enums import MerchantRole
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import (
    CustomerSubscriptionAlreadyCanceledError, CustomerSubscriptionNotEnabledError, CustomerSubscriptionNotFoundError,
    OrderNotFoundError
)
from mail.payments.payments.utils.datetime import utcnow


class CoreCancelCustomerSubscriptionAction(BaseDBAction):
    transact = True

    def __init__(self, uid: int, customer_subscription_id: int):
        super().__init__()
        self.uid: int = uid
        self.customer_subscription_id: int = customer_subscription_id

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        try:
            order: Order = await GetOrderByCustomerSubscriptionIdAction(
                uid=self.uid,
                customer_subscription_id=self.customer_subscription_id,
                for_update=True,
            ).run()
            assert order.shop is not None
        except OrderNotFoundError:
            raise CustomerSubscriptionNotFoundError

        assert order.customer_subscription and order.items

        if order.customer_subscription.time_finish is not None:
            raise CustomerSubscriptionAlreadyCanceledError

        if not order.customer_subscription.enabled:
            raise CustomerSubscriptionNotEnabledError

        time_finish = utcnow()
        acquirer = await GetAcquirerMerchantAction(uid=order.uid).run()
        acquirer = order.get_acquirer(acquirer)

        trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)
        await trust.subscription_cancel(
            uid=order.uid,
            acquirer=acquirer,
            order=order,
            item=order.single_item,
            finish_ts=time_finish,
            customer_uid=order.customer_uid,
        )

        order.customer_subscription.time_finish = time_finish
        order.customer_subscription.enabled = False
        order.customer_subscription = await self.storage.customer_subscription.save(order.customer_subscription)

        return order.customer_subscription, order


class CancelCustomerSubscriptionAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)
    transact = True

    def __init__(self, uid: int, customer_subscription_id: int):
        super().__init__()
        self.uid: int = uid
        self.customer_subscription_id: int = customer_subscription_id

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        return await CoreCancelCustomerSubscriptionAction(
            uid=self.uid,
            customer_subscription_id=self.customer_subscription_id,
        ).run()


class CancelCustomerSubscriptionServiceMerchantAction(AuthServiceMerchantMixin):
    transact = True

    def __init__(self, service_tvm_id: int, service_merchant_id: int, customer_subscription_id: int,
                 uid: Optional[int] = None):
        super().__init__()
        self.service_tvm_id: int = service_tvm_id
        self.service_merchant_id: int = service_merchant_id
        self.customer_subscription_id: int = customer_subscription_id
        self.uid: Optional[int] = uid

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        assert self.uid
        return await CoreCancelCustomerSubscriptionAction(
            uid=self.uid,
            customer_subscription_id=self.customer_subscription_id,
        ).run()
