from typing import Dict, Optional, Tuple

from sendr_interactions import exceptions as interaction_errors

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.create_or_update import CoreCreateOrUpdateOrderAction
from mail.payments.payments.core.actions.subscription.base import BaseSubscriptionAction
from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.enums import MerchantRole, OrderSource, ShopType
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import SubscriptionHasNoPriceForRegionId, UnknownRegionId
from mail.payments.payments.utils.helpers import without_none


class CoreCreateCustomerSubscriptionAction(BaseSubscriptionAction):
    subscription_check_moderation_approved = not settings.SUBSCRIPTION_MODERATION_DISABLED
    transact = True

    def __init__(self,
                 uid: int,
                 subscription_id: int,
                 quantity: int,
                 source: OrderSource,
                 customer_uid: Optional[int] = None,
                 user_ip: Optional[str] = None,
                 region_id: Optional[int] = None,
                 paymethod_id: Optional[str] = None,
                 service_merchant_id: Optional[int] = None,
                 service_client_id: Optional[int] = None,
                 create_order_extra: Optional[Dict] = None,
                 update_log_extra: Optional[Dict] = None,
                 ):
        super().__init__(uid=uid, subscription_id=subscription_id)
        self.quantity: int = quantity
        self.source = source
        self.customer_uid: Optional[int] = customer_uid
        self.user_ip: Optional[str] = user_ip
        self.region_id: Optional[int] = region_id
        self.paymethod_id: Optional[str] = paymethod_id
        self.service_merchant_id: Optional[int] = service_merchant_id
        self.service_client_id: Optional[int] = service_client_id
        self.create_order_extra = create_order_extra or {}
        self.update_log_extra = update_log_extra or {}

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        assert self.uid and self.subscription
        assert self.region_id or self.user_ip
        field = 'user_ip' if self.region_id is None else 'region_id'

        try:
            if self.region_id is None:
                assert self.user_ip
                self.region_id = await self.clients.geobase.get_region_id_by_ip(self.user_ip)
            parents_regions = await self.clients.geobase.get_parents(self.region_id)
        except interaction_errors.InteractionResponseError as exc:
            raise UnknownRegionId(params={"error": exc.params['error'], "field": field})

        price_by_region_id = {price.region_id: price for price in self.subscription.prices}
        price = None

        for region_id in parents_regions:
            if region_id in price_by_region_id:
                price = price_by_region_id[region_id]
                break

        if price is None:
            params = without_none({'region_id': self.region_id, 'user_ip': self.user_ip})
            raise SubscriptionHasNoPriceForRegionId(params=params)

        customer_subscription = CustomerSubscription(uid=self.uid,
                                                     service_merchant_id=self.service_merchant_id,
                                                     service_client_id=self.service_client_id,
                                                     subscription_id=self.subscription_id,
                                                     quantity=self.quantity,
                                                     user_ip=self.user_ip,
                                                     region_id=self.region_id)
        customer_subscription = await self.storage.customer_subscription.create(customer_subscription)

        order: Order = await CoreCreateOrUpdateOrderAction(
            uid=self.uid,
            source=self.source,
            create_order_extra=self.create_order_extra,
            update_log_extra=self.update_log_extra,
            caption=self.subscription.title,
            customer_subscription_id=customer_subscription.customer_subscription_id,
            autoclear=True,
            items=[{
                'amount': self.quantity,
                'name': self.subscription.title,
                'price': price.price,
                'nds': self.subscription.nds,
                'currency': price.currency
            }],
            paymethod_id=self.paymethod_id,
            default_shop_type=ShopType.from_oauth_mode(self.subscription.merchant_oauth_mode),
            acquirer=self.subscription.acquirer,
        ).run()

        customer_subscription.order_id = order.order_id
        customer_subscription = await self.storage.customer_subscription.save(customer_subscription)

        return customer_subscription, order


class CreateCustomerSubscriptionAction(BaseDBAction):
    required_merchant_roles = (MerchantRole.OPERATOR,)

    def __init__(self,
                 uid: int,
                 subscription_id: int,
                 quantity: int,
                 customer_uid: Optional[int] = None,
                 user_ip: Optional[str] = None,
                 region_id: Optional[int] = None,
                 paymethod_id: Optional[str] = None,
                 ):
        super().__init__()
        self.uid: int = uid
        self.subscription_id: int = subscription_id
        self.quantity: int = quantity
        self.customer_uid: Optional[int] = customer_uid
        self.user_ip: Optional[str] = user_ip
        self.region_id: Optional[int] = region_id
        self.paymethod_id: Optional[str] = paymethod_id

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        return await CoreCreateCustomerSubscriptionAction(uid=self.uid,
                                                          subscription_id=self.subscription_id,
                                                          quantity=self.quantity,
                                                          source=OrderSource.UI,
                                                          customer_uid=self.customer_uid,
                                                          user_ip=self.user_ip,
                                                          region_id=self.region_id,
                                                          paymethod_id=self.paymethod_id
                                                          ).run()


class CreateCustomerSubscriptionServiceMerchantAction(AuthServiceMerchantMixin):
    def __init__(self,
                 service_merchant_id: int,
                 service_tvm_id: int,
                 subscription_id: int,
                 quantity: int,
                 customer_uid: Optional[int] = None,
                 user_ip: Optional[str] = None,
                 region_id: Optional[int] = None,
                 paymethod_id: Optional[str] = None,
                 ):
        super().__init__()
        self.service_merchant_id: int = service_merchant_id
        self.service_tvm_id: int = service_tvm_id
        self.subscription_id: int = subscription_id
        self.quantity: int = quantity
        self.customer_uid: Optional[int] = customer_uid
        self.user_ip: Optional[str] = user_ip
        self.region_id: Optional[int] = region_id
        self.paymethod_id: Optional[str] = paymethod_id

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        assert self.uid

        create_order_extra = {
            'service_client_id': self.service_client_id,
            'service_merchant_id': self.service_merchant_id
        }
        update_log_extra = {
            'service_id': self.service.service_id if self.service else None,
            'service_name': self.service.name if self.service else None
        }

        return await CoreCreateCustomerSubscriptionAction(
            uid=self.uid,
            subscription_id=self.subscription_id,
            quantity=self.quantity,
            source=OrderSource.SERVICE,
            customer_uid=self.customer_uid,
            user_ip=self.user_ip,
            region_id=self.region_id,
            paymethod_id=self.paymethod_id,
            service_merchant_id=self.service_merchant_id,
            service_client_id=self.service_client_id,
            create_order_extra=create_order_extra,
            update_log_extra=update_log_extra,
        ).run()
