from typing import Optional, Tuple

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.mixins.auth_service_merchant import AuthServiceMerchantMixin
from mail.payments.payments.core.actions.order.get import GetOrderByCustomerSubscriptionIdAction
from mail.payments.payments.core.entities.customer_subscription import CustomerSubscription
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.entities.service import Service
from mail.payments.payments.core.exceptions import CustomerSubscriptionNotFoundError, OrderNotFoundError
from mail.payments.payments.utils.datetime import utcnow


class GetCustomerSubscriptionServiceMerchantAction(AuthServiceMerchantMixin, BaseDBAction):
    transact = True

    def __init__(self,
                 service_tvm_id: int,
                 service_merchant_id: int,
                 customer_subscription_id: int,
                 uid: Optional[int] = None,
                 service: Optional[Service] = None,
                 ):
        super().__init__()
        self.service_tvm_id: int = service_tvm_id
        self.service_merchant_id: int = service_merchant_id
        self.customer_subscription_id: int = customer_subscription_id
        self.uid: Optional[int] = uid
        self.service: Optional[Service] = service

    async def handle(self) -> Tuple[CustomerSubscription, Order]:
        assert self.uid

        try:
            order: Order = await GetOrderByCustomerSubscriptionIdAction(
                uid=self.uid,
                customer_subscription_id=self.customer_subscription_id,
            ).run()
        except OrderNotFoundError:
            raise CustomerSubscriptionNotFoundError

        assert order.customer_subscription
        customer_subscription = order.customer_subscription
        assert customer_subscription.customer_subscription_id

        if customer_subscription.time_finish is not None and customer_subscription.time_finish < utcnow():
            if customer_subscription.enabled:
                customer_subscription = await self.storage.customer_subscription.get(
                    uid=customer_subscription.uid,
                    customer_subscription_id=customer_subscription.customer_subscription_id,
                    for_update=True
                )
                customer_subscription.enabled = False
                customer_subscription = await self.storage.customer_subscription.save(customer_subscription)

        return customer_subscription, order
