from datetime import datetime, timezone
from typing import Any, Dict, Optional

from mail.payments.payments.core.actions.base.db import BaseDBAction
from mail.payments.payments.core.actions.interactions.trust import (
    ClearPaymentInTrustAction, GetPaymentInfoInTrustAction, GetSubscriptionInTrustAction
)
from mail.payments.payments.core.actions.merchant.get_acquirer import GetAcquirerMerchantAction
from mail.payments.payments.core.actions.mixins.callback_task import APICallbackTaskMixin
from mail.payments.payments.core.actions.order.get import GetOrderAction
from mail.payments.payments.core.actions.tlog.customer_subscription import ExportCustomerSubscriptionToTLogAction
from mail.payments.payments.core.entities.customer_subscription_transaction import CustomerSubscriptionTransaction
from mail.payments.payments.core.entities.enums import TransactionStatus
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.utils.datetime import utcnow


class SyncCustomerSubscriptionAction(APICallbackTaskMixin, BaseDBAction):
    transact = True

    def __init__(self, uid: int, order_id: int):
        super().__init__()
        self.uid: int = uid
        self.order_id: int = order_id

    def _extract_trust_order_id(self, payment_info: Dict[str, Any]) -> Optional[str]:
        orders = payment_info.get('orders', list())
        if len(orders) > 0:
            return orders[0].get('order_id', None)
        else:
            return None

    async def handle(self) -> None:
        order: Order = await GetOrderAction(
            uid=self.uid,
            order_id=self.order_id,
            select_customer_subscription=True,
            skip_add_crypto=True,
            with_customer_subscription=True,
            for_update=True,
        ).run()

        customer_subscription = order.customer_subscription
        assert customer_subscription and customer_subscription.customer_subscription_id
        customer_subscription_id = customer_subscription.customer_subscription_id

        acquirer = await GetAcquirerMerchantAction(uid=self.uid).run()
        acquirer = order.get_acquirer(acquirer)

        trust_subscription: Dict[str, Any] = await GetSubscriptionInTrustAction(order=order, acquirer=acquirer).run()

        purchase_tokens = set(trust_subscription['payments'])  # все платежи по подписке
        subs_until_ts = float(trust_subscription.get('subs_until_ts', 0))
        finish_ts = None
        if 'finish_ts' in trust_subscription:
            finish_ts = float(trust_subscription['finish_ts'])

        stored_final_purchase_tokens = set([
            purchase_token async for purchase_token in
            self.storage.customer_subscription_transaction.find_final_purchase_tokens(
                order.uid, customer_subscription_id
            )
        ])  # все платежи, на которые мы смотрели и они с финальными статусами

        for purchase_token in purchase_tokens.difference(stored_final_purchase_tokens):
            # Получаем статус транзакции
            payment_info: Dict[str, Any] = await GetPaymentInfoInTrustAction(
                purchase_token=purchase_token,
                order=order,
                acquirer=acquirer,
            ).run()
            payment_status = TransactionStatus.from_trust(payment_info['payment_status'])

            # Если заморожена
            if payment_status == TransactionStatus.HELD:
                # Клирим
                await ClearPaymentInTrustAction(purchase_token=purchase_token, order=order, acquirer=acquirer).run()
            # Если поклирилась с момента прошлого опроса
            elif payment_status == TransactionStatus.CLEARED:
                # Отправляем в nb
                await ExportCustomerSubscriptionToTLogAction(
                    uid=self.uid,
                    customer_subscription_id=customer_subscription_id,
                    purchase_token=purchase_token
                ).run_async()

            await self.storage.customer_subscription_transaction.create_or_update(
                CustomerSubscriptionTransaction(
                    uid=self.uid,
                    customer_subscription_id=customer_subscription_id,
                    purchase_token=purchase_token,
                    payment_status=payment_status,
                    data=payment_info,
                    trust_order_id=self._extract_trust_order_id(payment_info)
                )
            )

        old_enabled = customer_subscription.enabled
        new_time_until = datetime.fromtimestamp(subs_until_ts, tz=timezone.utc)
        is_prolongated = customer_subscription.time_until < new_time_until

        customer_subscription.time_until = new_time_until
        customer_subscription.enabled = utcnow() <= customer_subscription.time_until
        if finish_ts is not None:
            customer_subscription.time_finish = datetime.fromtimestamp(finish_ts, tz=timezone.utc)
            customer_subscription.enabled = utcnow() <= customer_subscription.time_finish

        customer_subscription = await self.storage.customer_subscription.save(customer_subscription)
        is_stopped = old_enabled != customer_subscription.enabled and customer_subscription.enabled is False

        if customer_subscription.service_client_id and customer_subscription.service_merchant_id:
            if is_prolongated or is_stopped:
                service = await self.storage.service.get_by_related(
                    service_client_id=customer_subscription.service_client_id,
                    service_merchant_id=customer_subscription.service_merchant_id,
                )
                if is_prolongated:
                    await self.create_customer_subscription_prolongated_service_task(customer_subscription, service)
                if is_stopped:
                    await self.create_customer_subscription_stopped_service_task(customer_subscription, service)
