import io
from itertools import filterfalse
from typing import AsyncIterable, List, Optional, Tuple

from aiohttp.multipart import BodyPartReader, MultipartReader
from PIL import Image

from sendr_utils import json_value

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog
from mail.payments.payments.core.entities.document import Document
from mail.payments.payments.core.entities.enums import DocumentType, MerchantRole, OperationKind
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import (
    CoreFailError, DocumentBodyPartError, DocumentCannotLoadImageError, DocumentFileSizeLimitExceededError,
    DocumentFileTypeNotAllowedError, DocumentNotFoundError, DocumentRequestBodyEmptyError, DocumentTypeNotAllowedError
)


class GetDocumentsAction(BaseMerchantAction):
    skip_data = True
    skip_parent = True
    required_merchant_roles = (MerchantRole.ADMIN,)

    async def handle(self) -> List[Document]:
        assert self.merchant
        return self.merchant.documents


class DeleteDocumentAction(BaseMerchantAction):
    transact = True
    skip_data = True
    skip_parent = True
    for_update = True
    required_merchant_roles = (MerchantRole.ADMIN,)

    check_moderation_disapproved = True

    def __init__(self,
                 path: str,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.path: str = path

    def delete_predicate(self, d: Document) -> bool:
        return d.path == self.path

    async def handle(self) -> None:
        assert self.merchant
        # Getting document for delete
        for_delete = list(filter(self.delete_predicate, self.merchant.documents))
        if not for_delete:
            raise DocumentNotFoundError(uid=self.merchant.uid, path=self.path)
        elif len(for_delete) > 1:
            self.logger.context_push(path=self.path)
            self.logger.error('Merchant has documents with the same path!')
            raise CoreFailError
        document: Document = for_delete[0]

        # Removing document from MDS
        await self.clients.payments_mds.remove(document.path)

        # Removing document from merchant
        self.merchant.documents = list(filterfalse(self.delete_predicate, self.merchant.documents))
        self.merchant = await self.storage.merchant.save(self.merchant)
        self.logger.context_push(revision=self.merchant.revision)
        self.logger.info('Merchant update: documents')

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=OperationKind.EDIT_MERCHANT,
            arguments={'delete_document': json_value(document)},
        ))


class DownloadDocumentAction(BaseMerchantAction):
    skip_data = True
    skip_parent = True
    required_merchant_roles = (MerchantRole.ADMIN,)

    def __init__(self,
                 path: str,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.path: str = path

    async def handle(self) -> Tuple[dict, AsyncIterable[bytes]]:
        assert self.merchant
        if self.merchant.get_document(self.path) is None:
            raise DocumentNotFoundError(uid=self.merchant.uid, path=self.path)

        content_type, data = await self.clients.payments_mds.download(self.path, close=True)
        extension = dict(settings.MIMETYPE_EXTENSION_MAPPING).get(content_type, '')
        content_disposition = f'attachment; filename="document{extension}"'
        headers = {
            'Content-Type': content_type,
            'Content-Disposition': content_disposition,
        }
        return headers, data


class UploadDocumentAction(BaseMerchantAction):
    transact = True
    skip_data = True
    skip_parent = True
    for_update = True
    required_merchant_roles = (MerchantRole.ADMIN,)

    check_moderation_disapproved = True

    def __init__(self,
                 reader: MultipartReader,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.reader: MultipartReader = reader

    @staticmethod
    async def _get_body_part(reader: MultipartReader) -> BodyPartReader:
        try:
            part = await reader.next()
        except ValueError:
            raise DocumentBodyPartError
        if part is None:
            raise DocumentRequestBodyEmptyError
        try:
            DocumentType(part.name)
        except ValueError:
            raise DocumentTypeNotAllowedError
        return part

    @staticmethod
    async def _read_body_part(part: BodyPartReader) -> Tuple[bytes, int]:
        size = 0
        chunks = []
        while True:
            chunk = await part.read_chunk()
            if not chunk:
                break
            size += len(chunk)
            if size > settings.DOCUMENT_MAX_SIZE:
                raise DocumentFileSizeLimitExceededError
            chunks.append(chunk)
        return b''.join(chunks), size

    @staticmethod
    def validate_image(data: bytes) -> None:
        """
        Checks that data is a valid image.
        """
        try:
            image = Image.open(io.BytesIO(data))
        except IOError:
            raise DocumentCannotLoadImageError
        mimetype = image.get_format_mimetype()
        if mimetype not in settings.ALLOWED_IMAGE_MIMETYPES:
            raise DocumentFileTypeNotAllowedError

    async def read(self) -> Tuple[DocumentType, bytes, int, str]:
        part = await self._get_body_part(self.reader)
        name = part.filename
        data, size = await self._read_body_part(part)
        return DocumentType(part.name), data, size, name

    async def handle(self) -> Document:
        assert self.merchant
        # Reading request data
        document_type, data, size, name = await self.read()
        self.validate_image(data)

        # Creating document instance and uploading it to mds
        path = await self.clients.payments_mds.upload(f'uid_{self.merchant.uid}.document', data)
        document = Document(
            document_type=document_type,
            path=path,
            size=size,
            name=name,
        )
        self.logger.context_push(path=path)

        # Adding document to merchant
        self.merchant.documents.append(document)
        self.merchant = await self.storage.merchant.save(self.merchant)
        self.logger.context_push(revision=self.merchant.revision)
        self.logger.info('Merchant update: documents')

        await self.storage.change_log.create(ChangeLog(
            uid=self.merchant.uid,
            revision=self.merchant.revision,
            operation=OperationKind.EDIT_MERCHANT,
            arguments={'upload_document': json_value(document)},
        ))
        return document
