from typing import Optional

from marshmallow.validate import Email

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.client import Client
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import CoreDataError, CoreFieldError
from mail.payments.payments.interactions.balance.entities import Person
from mail.payments.payments.interactions.balance.exceptions import BalanceDataError, BalanceWrongEmailError


class InitClientAction(BaseMerchantAction):
    transact = True
    for_update = True
    skip_parent = False
    skip_data = False

    def __init__(self, uid: Optional[int] = None, merchant: Optional[Merchant] = None, save: bool = True):
        super().__init__(uid=uid, merchant=merchant)
        self.save = save

    @staticmethod
    def person_data(merchant: Merchant) -> dict:
        assert (
            merchant.addresses is not None
            and merchant.bank is not None
            and merchant.organization is not None
            and merchant.ceo is not None
        )
        address_by_type = {address.type: address for address in merchant.addresses}
        legal = address_by_type['legal']
        addresses = {
            'legal_address_city': legal.city,
            'legal_address_home': legal.home,
            'legal_address_postcode': legal.zip,
            'legal_address_street': legal.street,
        }
        if 'post' in address_by_type:
            post = address_by_type['post']
            addresses.update({
                'address_city': post.city,
                'address_home': post.home,
                'address_postcode': post.zip,
                'address_street': post.street,
            })
        return {
            'account': merchant.bank.account,
            'bik': merchant.bank.bik,
            'fname': merchant.ceo.name,
            'lname': merchant.ceo.surname,
            'mname': merchant.ceo.patronymic,
            'email': merchant.ceo.email,
            'phone': merchant.ceo.phone,

            'name': merchant.organization.name,
            'longname': merchant.organization.full_name,
            'inn': merchant.organization.inn,
            'kpp': merchant.organization.kpp,
            'ogrn': merchant.organization.ogrn,

            **addresses,
        }

    async def _init_client(self, merchant: Merchant) -> None:
        assert merchant.ceo

        try:
            # Creating or updating client
            client = await self.clients.balance.find_client(merchant.uid)
            client_id = await self.clients.balance.create_client(
                uid=merchant.uid,
                client=Client(
                    client_id=client.client_id if client else None,  # type: ignore
                    name=f'{merchant.ceo.surname} {merchant.ceo.name}',
                    email=merchant.ceo.email,
                    phone=merchant.ceo.phone,
                ),
            )

            # If client didn't exist before, linking it to uid
            if client is None:
                await self.clients.balance.create_user_client_association(merchant.uid, client_id)
            self.logger.context_push(client_id=client_id)

            # Creating or updating person
            person = Person(
                client_id=client_id,
                person_id=merchant.person_id,
                **self.person_data(merchant),
            )
            person_id = await self.clients.balance.create_person(merchant.uid, person)
            self.logger.context_push(person_id=person_id)

            # Updating client_id, person_id on entity
            merchant.client_id = client_id
            merchant.person_id = person_id
            self.logger.info('Balance client initialized')
        except BalanceWrongEmailError as exc:
            self.logger.context_push(balance_error=exc.MESSAGE)
            self.logger.info('Balance client initialization failed: BalanceWrongEmailError')
            raise CoreFieldError(
                exc.MESSAGE,
                fields={
                    "persons.ceo.email": Email.default_message
                }
            )
        except BalanceDataError as exc:
            self.logger.context_push(balance_error=exc.MESSAGE)
            self.logger.info('Balance client initialization failed: BalanceDataError')
            raise CoreDataError(exc.MESSAGE)

    async def handle(self) -> Merchant:
        # Initializing client
        await self._init_client(self.merchant)

        # Saving if needed
        if self.save:
            self.merchant: Merchant = await self.storage.merchant.save(self.merchant)
            await self.storage.change_log.create(ChangeLog(
                uid=self.merchant.uid,
                revision=self.merchant.revision,
                operation=OperationKind.EDIT_MERCHANT,
                arguments={
                    'client_id': self.merchant.client_id,
                    'person_id': self.merchant.person_id,
                },
            ))
            self.logger.context_push(revision=self.merchant.revision)
            self.logger.info('Merchant updated: client_id, person_id')

        return self.merchant
