from typing import Optional

from mail.payments.payments.core.actions.base.merchant import BaseMerchantAction
from mail.payments.payments.core.entities.change_log import ChangeLog, OperationKind
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.exceptions import TinkoffMerchantInvalidAddressError, TinkoffMerchantInvalidDataError
from mail.payments.payments.interactions.exceptions import TinkoffAddressError, TinkoffError


class InitSubmerchantAction(BaseMerchantAction):
    for_update = True
    skip_moderation = True

    def __init__(self,
                 uid: Optional[int] = None,
                 merchant: Optional[Merchant] = None,
                 save: bool = False,
                 ):
        super().__init__(uid=uid, merchant=merchant)
        self.save = save

    async def handle(self) -> Merchant:
        assert self.merchant
        try:
            submerchant_id = await self.clients.tinkoff.create_merchant(self.merchant)
            self.merchant.submerchant_id = submerchant_id

            self.logger.context_push(submerchant_id=submerchant_id)
            self.logger.info('Submerchant initialized')
        except TinkoffAddressError as error:
            self.logger.info('Tinkoff address error')
            raise TinkoffMerchantInvalidAddressError(params=error.params)
        except TinkoffError as error:
            self.logger.info('Tinkoff error')
            raise TinkoffMerchantInvalidDataError(params=error.params)

        # Saving if needed
        if self.save:
            async with self.storage.conn.begin():
                self.merchant = await self.storage.merchant.save(self.merchant)
                await self.storage.change_log.create(ChangeLog(
                    uid=self.merchant.uid,
                    revision=self.merchant.revision,
                    operation=OperationKind.EDIT_MERCHANT,
                    arguments={'submerchant_id': submerchant_id},
                ))
                self.logger.info('Merchant updated: submerchant_id')

        return self.merchant
