import asyncio
from typing import Any, Dict, List, Optional, Tuple

from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.utils.helpers import md5_hex


class SenderLetterIsSpamAction(BaseAction):
    def __init__(self,
                 mailing_id: str,
                 to_email: str,
                 render_context: Dict,
                 user_ip: str,
                 from_email: str,
                 from_uid: int,
                 ):
        super().__init__()
        self.mailing_id: str = mailing_id
        self.to_email: str = to_email
        self.render_context: Dict = render_context
        self.user_ip: str = user_ip
        self.from_email: str = from_email
        self.from_uid: int = from_uid

    @classmethod
    def _flat_context(cls,
                      context: Any,
                      key_prefix: str = '',
                      key_suffix: str = '',
                      hash_keys: bool = False,
                      prefix: Optional[List[str]] = None) -> Tuple[Any, Dict[str, str]]:
        # https://wiki.yandex-team.ru/antispam/sonline/moduli/checkform/#proverkaform2.0
        # По данному контексту необходимо отрендерить шаблон в рассыляторе с плейсхолдерами, затем шаблон с реальными
        # данными, после этого передать в SO оба шаблона и словарь соответствий плейсхолдеров и значений
        # Для понимания работы функции лучше всего посмотреть тест:
        # payments/tests/core/actions/test_so.py
        fields = dict()
        fields_context = type(context)()

        prefix = prefix or []

        if isinstance(context, dict):
            for key, value in context.items():
                new_value, fields_part = cls._flat_context(value,
                                                           key_prefix=key_prefix,
                                                           key_suffix=key_suffix,
                                                           prefix=prefix + [key],
                                                           hash_keys=hash_keys)
                fields_context[key] = new_value
                fields.update(fields_part)
        elif isinstance(context, list):
            for num, item in enumerate(context):
                new_value, fields_part = cls._flat_context(item,
                                                           key_prefix=key_prefix,
                                                           key_suffix=key_suffix,
                                                           prefix=prefix + [str(num)],
                                                           hash_keys=hash_keys)
                fields_context.append(new_value)
                fields.update(fields_part)
        else:
            key = '.'.join(prefix)
            if hash_keys:
                key = md5_hex(key)
            return f'{key_prefix}{key}{key_suffix}', {key: context}

        return fields_context, fields

    async def handle(self) -> bool:
        render_context = self.render_context

        fields_context, fields = self._flat_context(render_context, key_prefix='{', key_suffix='}', hash_keys=True)
        campaign = await self.clients.sender.campaign_detail(self.mailing_id)

        letter_ids = [letter['id'] for letter in campaign['letters']]
        campaign_id = campaign['id']

        # Рендеринг письма как шаблона для SO и как итогового письма
        body_templates, bodies = await asyncio.gather(
            asyncio.gather(*[
                self.clients.sender.render_transactional_letter(campaign_id, letter_id, render_context=fields_context)
                for letter_id in letter_ids
            ]),
            asyncio.gather(*[
                self.clients.sender.render_transactional_letter(campaign_id, letter_id, render_context=render_context)
                for letter_id in letter_ids
            ])
        )

        is_spam_statuses = await asyncio.gather(*[
            self.clients.so.form_is_spam(request_id=self.request_id,
                                         form_id=self.mailing_id,
                                         user_ip=self.user_ip,
                                         to_email=self.to_email,
                                         from_email=self.from_email,
                                         from_uid=self.from_uid,
                                         fields=fields,
                                         body_template=body_template,
                                         body=body)
            for body_template, body in zip(body_templates, bodies)
        ])

        return any((is_spam for is_spam in is_spam_statuses))
