from typing import Dict, Iterable, Optional, Tuple, Union

from sendr_utils import utcnow

from mail.payments.payments.conf import settings
from mail.payments.payments.core.actions.base.action import BaseAction
from mail.payments.payments.core.actions.get_oauth import GetMerchantOAuth
from mail.payments.payments.core.actions.order.get_trust_env import GetOrderTrustEnvAction
from mail.payments.payments.core.entities.enums import (
    AcquirerType, MerchantOAuthMode, PaymentsTestCase, PayStatus, TransactionStatus, TrustEnv
)
from mail.payments.payments.core.entities.item import Item
from mail.payments.payments.core.entities.merchant import Merchant
from mail.payments.payments.core.entities.order import Order
from mail.payments.payments.core.exceptions import (
    CoreActionDenyError, CustomerUidInvalid, OAuthAbsentError, SubscriptionRequiresCustomerUid,
    TinkoffInvalidSubmerchantIdError
)
from mail.payments.payments.interactions.trust.entities import PaymentMode
from mail.payments.payments.interactions.trust.exceptions import TrustOrderAlreadyExists, TrustUidNotFound


class BaseTrustAction(BaseAction):
    def __init__(self, purchase_token: str, order: Order, acquirer: AcquirerType):
        assert order.shop is not None
        super().__init__()
        self.purchase_token: str = purchase_token
        self.order: Order = order
        self.acquirer: AcquirerType = acquirer


class GetTrustCredentialParamsAction(BaseAction):
    """
    Определяем с каким эквайером и дополнительными параметрами нужно взаимодействовать с API Траста
    для проведения оплаты для мерчанта.

    В принципе в сам Траст передавать эквайер не нужно: нам приходится тут отдавать эквайера, так
    как наш клиент к АПИ Траста по эквайеру определяет, передать ли в АПИ
    ключик OAuth токен доступа в "магазин" Яндекс.Кассы или же идентификатор "магазина" в Тинькове.

    Тестовые платежи в продакшен окружении проводятся через специальное "сендбокс" окружение Траста:
    интеграционная тестовая среда trust-payments-test.paysys.yandex.net
    https://wiki.yandex-team.ru/TRUST/Payments/#sredyidostupy.

    Стоит заметить, что в Яндекс.Кассе есть разделение на продовые и тестовые магазины, т. е.
    можно передать токен тестового магазина в сендбокс Траст.

    Однако мы не требуем строгого наличия
    токена тестового магазина Яндекс.Кассы - для сендбокс окружение Траста в случае отсутствия
    тестового OAuth токена можно проводить тестовый платеж как-будто бы через Тинькова.

    Также мы не требуем регистрации мерчанта в Тинькове, если проводится тестовый платеж:
    можно сходить в тесовый интеграционный Траст с дефолтным идентификатором "магазина" из Тинькова.

    Иначе говоря:
    если проводится боевой платеж (order.shop.shop_type == ShopType.PROD), то данный экшен проверит, что
        1) если AcquirerType.KASSA, то для мерчанта есть submerchant_id
           вообще во время регистрации мерчанта мы всегда проставляем submerchant_id, просто при проведении платежа
           через Кассу он формально не нужен, и поэтому мы не бросим ошибку, если его вдруг не будет
        2) если AcquirerType.KASSA, то есть OAuth токен для доступа в магазин Янденкс.Кассы
    """

    def __init__(self, acquirer: Optional[AcquirerType], order: Order, merchant: Merchant):
        # Возможно имеет смысл не принимать явно acquirer, а брать
        # order.get_acquirer(merchant.acquirer) с ассертом, что результат не None?
        assert order.shop is not None
        super().__init__()
        self.order = order
        self.merchant = merchant
        self.acquirer = acquirer

    async def handle(self) -> Tuple[AcquirerType, Optional[str], Optional[MerchantOAuthMode]]:
        assert self.order.shop is not None
        trust_env: TrustEnv = await GetOrderTrustEnvAction(self.order).run()
        acquirer = self.acquirer

        try:
            submerchant_id = self.merchant.get_submerchant_id()
        except TinkoffInvalidSubmerchantIdError as e:
            # если мы тут, то эквайер Тиньков, а submerchant_id нету
            # например, пререгистрированный продавец без эквайера?
            if trust_env != TrustEnv.SANDBOX:
                raise e
            submerchant_id = settings.TRUST_SANDBOX_SUBMERCHANT_ID

        try:
            assert self.order.shop is not None
            oauth = await GetMerchantOAuth(acquirer=acquirer, merchant=self.merchant, shop=self.order.shop).run()
        except OAuthAbsentError as e:
            # если мы тут, то эквайер Касса, но токена нет
            if trust_env != TrustEnv.SANDBOX:
                raise e
            acquirer = AcquirerType.TINKOFF
            oauth = None

        if acquirer is None:
            if trust_env != TrustEnv.SANDBOX:
                raise CoreActionDenyError
            acquirer = AcquirerType.TINKOFF

        return acquirer, submerchant_id, oauth


class ProcessNewOrderInTrustAction(BaseAction):
    def __init__(self,
                 merchant: Merchant,
                 order: Order,
                 items: Iterable[Item],
                 return_url: Optional[str],
                 trust_form_name: Optional[str] = None,
                 template_tag: str = 'mobile/form',
                 yandexuid: Optional[str] = None,
                 customer_uid: Optional[int] = None,
                 user_email: Optional[str] = None,
                 paymethod_id: Optional[str] = 'trust_web_page',
                 payment_mode: Optional[PaymentMode] = PaymentMode.WEB_PAYMENT,
                 selected_card_id: Optional[str] = None,
                 service_fee: Optional[int] = None,
                 commission: Optional[int] = None,
                 payment_completion_action: Optional[Union[str, Dict[str, str]]] = None,
                 payments_service_slug: Optional[str] = None,
                 login_id: Optional[str] = None,
                 ):
        assert order.shop is not None
        super().__init__()
        self.merchant = merchant
        self.order = order
        self.items = list(items)
        self.return_url = return_url
        self.trust_form_name = trust_form_name
        self.template_tag = template_tag
        self.yandexuid = yandexuid
        self.customer_uid = customer_uid
        self.user_email = user_email
        self.paymethod_id = paymethod_id
        self.payment_mode = payment_mode
        self.selected_card_id = selected_card_id
        self.service_fee = service_fee
        self.commission = commission
        self.payment_completion_action = payment_completion_action
        self.payments_service_slug = payments_service_slug
        self.login_id = login_id

    async def handle(self) -> Dict:
        assert self.order.shop
        if self.order.test is not None:
            return {
                'purchase_token': settings.TEST_PURCHASE_TOKEN,
                'payment_url': settings.TEST_PAYMENT_URL,
                'trust_payment_id': settings.TEST_TRUST_PAYMENT_ID
            }

        if self.order.is_subscription:
            assert self.order.customer_subscription and self.order.customer_subscription.subscription
            if not self.customer_uid:
                raise SubscriptionRequiresCustomerUid

        trust = self.clients.get_trust_client(self.merchant.uid, self.order.shop.shop_type)

        acquirer, submerchant_id, oauth = await GetTrustCredentialParamsAction(
            acquirer=self.order.get_acquirer(self.merchant.acquirer),
            merchant=self.merchant,
            order=self.order
        ).run()

        try:
            if self.order.is_subscription:
                await trust.subscription_create(
                    uid=self.order.uid,
                    acquirer=acquirer,
                    order=self.order,
                    item=next(iter(self.items)),
                    customer_uid=self.customer_uid,
                )
            else:
                await trust.orders_create(
                    uid=self.merchant.uid,
                    acquirer=acquirer,
                    merchant=self.merchant,
                    order=self.order,
                    items=self.items,
                    customer_uid=self.customer_uid,
                    service_fee=self.service_fee,
                    commission=self.commission,
                )
        except TrustOrderAlreadyExists:
            self.logger.context_push(order_id=self.order.order_id, customer_uid=self.customer_uid)
            self.logger.warning('Order already exist')
        except TrustUidNotFound:
            raise CustomerUidInvalid

        payment_create_result = await trust.payment_create(
            uid=self.merchant.uid,
            order=self.order,
            items=self.items,
            return_url=self.return_url,
            template_tag=self.template_tag,
            user_email=self.user_email or '',
            yandexuid=self.yandexuid,
            payments_service_slug=self.payments_service_slug,
            customer_uid=self.customer_uid,
            trust_form_name=self.trust_form_name,
            payment_mode=self.payment_mode,
            paymethod_id=self.paymethod_id,
            selected_card_id=self.selected_card_id,
            payment_completion_action=self.payment_completion_action,
            login_id=self.login_id,
            acquirer=acquirer,
            submerchant_id=submerchant_id,
            oauth=oauth,
        )

        if any(item.markup for item in self.items):
            await trust.payment_markup(
                uid=self.merchant.uid,
                acquirer=acquirer,
                items=self.items,
                purchase_token=payment_create_result['purchase_token'],
                order=self.order,
                customer_uid=self.customer_uid,
            )

        result = await trust.payment_start(
            uid=self.merchant.uid,
            acquirer=acquirer,
            purchase_token=payment_create_result['purchase_token'],
            without_url=self.order.data.recurrent or self.order.data.without_3ds,
        )

        result['trust_payment_id'] = payment_create_result['trust_payment_id']
        return result


class GetSubscriptionInTrustAction(BaseAction):
    def __init__(self, order: Order, acquirer: AcquirerType):
        super().__init__()
        self.order: Order = order
        self.acquirer: AcquirerType = acquirer

    async def handle(self) -> dict:
        assert self.order.items
        order = self.order
        assert order.shop is not None
        trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)
        return await trust.subscription_get(
            uid=order.uid,
            acquirer=self.acquirer,
            order=order,
            item=order.single_item,
            customer_uid=order.customer_uid,
        )


class ClearPaymentInTrustAction(BaseTrustAction):
    async def handle(self) -> Dict:
        order = self.order
        uid = order.uid
        assert order.shop is not None
        trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)
        dummy_response = {"status": "success", "status_code": "payment_is_updated"}

        if order.test is not None:
            return dummy_response

        if trust.get_merchant_setting(uid, 'trust_skip_clear'):
            self.logger.context_push(uid=uid, order_id=order.order_id)
            self.logger.info('Clear skipped due to trust_skip_clear')
            return dummy_response

        return await trust.payment_clear(
            uid=order.uid,
            acquirer=self.acquirer,
            purchase_token=self.purchase_token,
        )


class UnholdPaymentInTrustAction(BaseTrustAction):
    async def handle(self) -> Dict:
        if self.order.test is not None:
            return {
                "status": "success",
                "status_code": "payment_is_updated"
            }
        assert self.order.shop is not None
        trust = self.clients.get_trust_client(self.order.uid, self.order.shop.shop_type)
        return await trust.payment_unhold(
            uid=self.order.uid,
            acquirer=self.acquirer,
            purchase_token=self.purchase_token,
        )


class GetPaymentStatusInTrustAction(BaseTrustAction):
    async def handle(self) -> str:
        order = self.order
        if order.test is not None:
            if self.order.test == PaymentsTestCase.TEST_OK_CLEAR:
                return TransactionStatus.to_trust(TransactionStatus.CLEARED)
            else:
                return TransactionStatus.to_trust(TransactionStatus.ACTIVE)
        assert order.shop is not None
        trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)
        return await trust.payment_status(
            uid=order.uid,
            acquirer=self.acquirer,
            purchase_token=self.purchase_token,
        )


class GetPaymentInfoInTrustAction(BaseTrustAction):
    async def handle(self) -> Dict:
        order = self.order
        if order.test is not None:
            if self.order.test == PaymentsTestCase.TEST_PAYMENT_FAILED:
                response = {'payment_status': TransactionStatus.to_trust(TransactionStatus.FAILED),
                            'payment_resp_code': '400',
                            'payment_resp_desc': 'invalid_order'}
            elif order.pay_status in {PayStatus.MODERATION_NEGATIVE, PayStatus.IN_CANCEL}:
                response = {'payment_status': TransactionStatus.to_trust(TransactionStatus.CANCELLED)}
            elif order.test == PaymentsTestCase.TEST_OK_CLEAR or order.pay_status == PayStatus.IN_PROGRESS:
                response = {'payment_status': TransactionStatus.to_trust(TransactionStatus.CLEARED)}
            else:
                response = {'payment_status': TransactionStatus.to_trust(TransactionStatus.HELD)}
            response.update({
                'uid': settings.TEST_CUSTOM_UID,
                'clear_real_ts': utcnow().timestamp()
            })
            return response

        assert order.shop is not None
        trust = self.clients.get_trust_client(order.uid, order.shop.shop_type)
        return await trust.payment_get(
            uid=order.uid,
            acquirer=self.acquirer,
            purchase_token=self.purchase_token,
            with_terminal_info=True
        )
