from typing import ClassVar, Dict, List, Optional
from uuid import UUID

from mail.payments.payments.core.actions.base.db import BaseAction
from mail.payments.payments.core.entities.document import Document
from mail.payments.payments.core.entities.enums import DocumentType
from mail.payments.payments.interactions.yandex_pay_admin.entities import Document as YandexPayAdminDocument
from mail.payments.payments.interactions.yandex_pay_admin.entities import YandexPayAdminDocumentType


class MerchantModerationResultNotifyAction(BaseAction):
    type_map: ClassVar[Dict[DocumentType, YandexPayAdminDocumentType]] = {
        DocumentType.OFFER: YandexPayAdminDocumentType.OFFER,
        DocumentType.PASSPORT: YandexPayAdminDocumentType.PASSPORT,
        DocumentType.PROXY: YandexPayAdminDocumentType.POWER_OF_ATTORNEY,
        DocumentType.SIGNER_PASSPORT: YandexPayAdminDocumentType.SIGNER_PASSPORT,
        DocumentType.OTHER: YandexPayAdminDocumentType.OTHER,
        DocumentType.PCI_DSS_CERT: YandexPayAdminDocumentType.PCI_DSS_CERT,
    }

    def __init__(self, partner_id: UUID, verified: Optional[bool], documents: List[Document]):
        super().__init__()
        self.partner_id = partner_id
        self.verified = verified
        self.documents = documents

    @classmethod
    def _make_document(cls, document: Document) -> YandexPayAdminDocument:
        return YandexPayAdminDocument(
            path=document.path,
            name=document.name,
            type=cls.type_map[document.document_type],
        )

    async def handle(self) -> None:
        await self.clients.yandex_pay_admin.update_moderation(
            partner_id=self.partner_id,
            verified=self.verified,
            documents=[
                self._make_document(document)
                for document in self.documents
            ],
        )
