from datetime import datetime
from typing import Any, ClassVar, Dict, List, Optional, Tuple

from sendr_utils import alist

from mail.payments.payments.core.actions.manager.base import BaseManagerAction
from mail.payments.payments.core.actions.merchant.get import GetMerchantAction
from mail.payments.payments.core.entities.common import SearchStats
from mail.payments.payments.core.entities.enums import AcquirerType, MerchantStatus, ModerationStatus, Role
from mail.payments.payments.core.entities.keyset import Keyset, KeysetEntry, ManagerAnalyticsKeysetEntity
from mail.payments.payments.core.entities.merchant import (
    Merchant, MerchantAnalyticsStats, MerchantsAnalyticsAdminData, MerchantWithAnalytics
)
from mail.payments.payments.core.exceptions import KeysetInvalidError, SortByInvalidError


class GetMerchantAnalyticsManagerAction(BaseManagerAction):
    require_roles = (Role.ASSESSOR,)

    SORT_BY_DEFAULT: str = 'uid'
    SORT_BY_ACCEPTABLE: ClassVar[Tuple[str, ...]] = ('uid', 'created', 'payments_success', 'money_success')

    def __init__(self,
                 manager_uid: int,
                 limit: Optional[int] = None,
                 sort_by: str = SORT_BY_DEFAULT,
                 desc: bool = False,
                 merchant_uid: Optional[int] = None,
                 name: Optional[str] = None,
                 moderation_status: Optional[ModerationStatus] = None,
                 acquirer: Optional[AcquirerType] = None,
                 blocked: Optional[bool] = None,
                 site_url: Optional[str] = None,
                 created_from: Optional[datetime] = None,
                 created_to: Optional[datetime] = None,
                 pay_created_from: Optional[datetime] = None,
                 pay_created_to: Optional[datetime] = None,
                 pay_closed_from: Optional[datetime] = None,
                 pay_closed_to: Optional[datetime] = None,
                 service_ids: Optional[List[int]] = None,
                 keyset: Optional[ManagerAnalyticsKeysetEntity] = None,
                 ):
        super().__init__(manager_uid=manager_uid)
        self.merchant_uid: Optional[int] = merchant_uid
        self.name: Optional[str] = name
        self.moderation_status: Optional[ModerationStatus] = moderation_status
        self.acquirer: Optional[AcquirerType] = acquirer
        self.blocked: Optional[bool] = blocked
        self.site_url: Optional[str] = site_url
        self.created_from: Optional[datetime] = created_from
        self.created_to: Optional[datetime] = created_to
        self.pay_created_from: Optional[datetime] = pay_created_from
        self.pay_created_to: Optional[datetime] = pay_created_to
        self.pay_closed_from: Optional[datetime] = pay_closed_from
        self.pay_closed_to: Optional[datetime] = pay_closed_to
        self.service_ids: Optional[List[int]] = service_ids
        self.keyset: Optional[ManagerAnalyticsKeysetEntity] = keyset
        self.limit: Optional[int] = limit
        self.sort_by: str = sort_by
        self.desc: bool = desc

    def _make_next_page_keyset(self, keyset: Optional[Keyset]) -> Optional[ManagerAnalyticsKeysetEntity]:
        if not keyset:
            return None
        sort_by = []
        keyset_entries = {}
        for column, order, barrier in keyset:
            sort_by.append((column, order))
            keyset_entry = KeysetEntry(order=order, barrier=barrier)

            keyset_entries[column] = keyset_entry

        return ManagerAnalyticsKeysetEntity(sort_order=[item[0] for item in sort_by], **keyset_entries)

    def _get_keyset_filter(self) -> Optional[Keyset]:
        if self.keyset is None:
            return None
        keyset: Keyset = []
        for name in self.keyset.sort_order:
            entry: KeysetEntry = getattr(self.keyset, name)
            if name != self.sort_by and name != 'uid':
                raise KeysetInvalidError

            keyset.append((name, entry.order, entry.barrier))

        return keyset

    async def _get_merchant_with_analytics(self,
                                           merchant: Merchant,
                                           stats: MerchantAnalyticsStats
                                           ) -> MerchantWithAnalytics:
        roles = self._with_subroles(
            [r.role async for r in self.storage.manager_role.find(manager_uid=self.manager_uid)]
        )
        if Role.ACCOUNTANT not in roles:
            stats = stats.without_money()

        service_merchants = await alist(self.storage.service_merchant.find(uid=merchant.uid, with_service=True))

        merchant = await GetMerchantAction(merchant=merchant).run()
        merchant_with_analytics = MerchantWithAnalytics(
            uid=merchant.uid,
            name=merchant.name,
            moderation=merchant.moderation,
            created=merchant.created,
            acquirer=merchant.acquirer,
            blocked=merchant.blocked,
            client_id=merchant.client_id,
            organization=merchant.organization,
            service_merchants=service_merchants,
            support_comment=merchant.support_comment,
            contact=merchant.contact,
            payments_total=stats.payments_total,
            payments_success=stats.payments_success,
            payments_refund=stats.payments_refund,
            money_success=stats.money_success,
            money_refund=stats.money_refund,
        )
        return merchant_with_analytics

    async def _get_merchants(self) -> Tuple[List[MerchantWithAnalytics], SearchStats, Optional[Keyset]]:
        filter_params: Dict[str, Any] = {
            'uid': self.merchant_uid,
            'name': self.name,
            'statuses': [_ for _ in MerchantStatus if _ != MerchantStatus.DRAFT],
            'moderation_status': self.moderation_status,
            'acquirer': self.acquirer,
            'blocked': self.blocked,
            'site_url': self.site_url,
            'created_from': self.created_from,
            'created_to': self.created_to,
            'pay_created_from': self.pay_created_from,
            'pay_created_to': self.pay_created_to,
            'pay_closed_from': self.pay_closed_from,
            'pay_closed_to': self.pay_closed_to,
            'service_ids': self.service_ids,
        }

        sort_by: Optional[str] = self.sort_by
        keyset = self._get_keyset_filter()
        if keyset is not None:
            sort_by = None

        merchants, keyset = await self.storage.merchant.get_analytics(
            limit=self.limit,
            sort_by=sort_by,
            descending=self.desc,
            keyset=keyset,
            **filter_params,
        )

        merchants_with_analytics = [
            await self._get_merchant_with_analytics(merchant, stats)
            for merchant, stats in merchants
        ]
        found = await self.storage.merchant.get_analytics_found(**filter_params)
        total = await self.storage.merchant.get_analytics_found(
            statuses=[_ for _ in MerchantStatus if _ != MerchantStatus.DRAFT],
        )
        return merchants_with_analytics, SearchStats(total=total, found=found), keyset

    async def handle(self) -> MerchantsAnalyticsAdminData:
        if self.sort_by not in self.SORT_BY_ACCEPTABLE:
            raise SortByInvalidError
        merchants, search_stats, keyset = await self._get_merchants()

        processed_keyset = self._make_next_page_keyset(keyset)

        return MerchantsAnalyticsAdminData(
            merchants=merchants,
            stats=search_stats,
            keyset=processed_keyset,
        )
